//-----------------------------------------------------------
// UT2DPawn
//
// Copyright 2008 Joe Bates
//
// This file is part of UT2D.
//
// UT2D is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// UT2D is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with UT2D.  If not, see <http://www.gnu.org/licenses/>.
//
// Last Modified: October 28, 2008
//-----------------------------------------------------------
class UT2DPawn extends UTPawn;

var rotator LastRotation;

var float LastTrans;

var float WallDodgeSpeed;
var float WallDodgeSpeedZ;

//Tick
simulated function Tick(float DeltaTime)
{
	local vector loc;
	local rotator rot;
	local int deltaPitch;
	local bool bTurned;

	super.Tick(DeltaTime);
	
	if(bBlockActors)
		SetCollision(bCollideActors, false);
	
	//Clamp to Y = 0.
	loc = Location;
	loc.Y = 0;
	SetLocation(loc);
	
	//Swimming/Flying Fix
	if(Physics == PHYS_Swimming || Physics == PHYS_Flying)
	{
		rot = Rotation;
		rot.Pitch = 0;
		SetRotation(rot);
	}

	//Bot Hacks
	if(UTBot(Controller) == none)
		return;
	
	rot = rotator(Controller.FocalPoint - (Location + vect(0, 0, 1) * HeadOffset));
	
	//Clamp to 2D
	if(abs(rot.Yaw) < 16384)
		rot.Yaw = 0;
	else
		rot.Yaw = 32768;
		
	deltaPitch = RotationRate.Pitch * DeltaTime * 3;
	
	//If it's looking at the side it wants to
	if(rot.Yaw == LastRotation.Yaw)
	{
		if(abs(rot.Pitch - LastRotation.Pitch) <= deltaPitch)
			LastRotation = rot; //Just look at an enemy if its close
		else if(rot.Pitch > LastRotation.Pitch) 
			LastRotation.Pitch += deltaPitch; //Move up towards it
		else
			LastRotation.Pitch -= deltaPitch; //Move down towards it
	}
	else
	{
		if(rot.Pitch >= 0)
			LastRotation.Pitch += deltaPitch; //Move up towards it
		else
			LastRotation.Pitch -= deltaPitch; //Move down towards it
	}
		
	//Turn around
	if(LastRotation.Pitch > 16384)
	{
		LastRotation.Pitch = 32768 - LastRotation.Pitch;
		bTurned = true;
	}
	else if(LastRotation.Pitch < -16384)
	{
		LastRotation.Pitch = -32768 - LastRotation.Pitch;
		bTurned = true;
	}
	
	if(bTurned)
	{
		if(abs(LastRotation.Yaw) < 16384)
			LastRotation.Yaw = 32768;
		else
			LastRotation.Yaw = 0;
	}
	
	Controller.SetRotation(LastRotation);
	
	rot = LastRotation;
	SetRemoteViewPitch(rot.Pitch);
	rot.Pitch = 0;
	SetRotation(rot);
}

//Bot Hack
function bool NeedToTurn(vector Target)
{
	return false;
}

//Bounce into 2D
simulated event HitWall(vector HitNormal, Actor Wall, PrimitiveComponent WallComp)
{
	if(HitNormal.X != 0)
	{
		HitNormal.Y = 0;
		HitNormal = normal(HitNormal);
	}
	
	super.HitWall(HitNormal, Wall, WallComp);
}

//AddVelocity
function AddVelocity(vector NewVelocity, vector HitLocation, class<DamageType> damageType, optional TraceHitInfo HitInfo)
{
	local float magnitude;
	
	//Clamp to 2D.
	magnitude = vsize(NewVelocity);
	NewVelocity.Y = 0;
	NewVelocity = normal(NewVelocity) * magnitude;
	
	super.AddVelocity(NewVelocity, HitLocation, damageType, HitInfo);
}

//UpdateEyeHeight
event UpdateEyeHeight(float DeltaTime)
{
	if(!bIsCrouched)
		super.UpdateEyeHeight(DeltaTime);
}

//Dodge
function bool Dodge(eDoubleClickDir DoubleClickMove)
{
	local vector TraceStart, TraceEnd, Dir, Cross, HitLocation, HitNormal;
	local Actor HitActor;
	local bool bFalling, out;

	if(bIsCrouched || bWantsToCrouch || (Physics != PHYS_Walking && Physics != PHYS_Falling))
		return false;

	if(Physics == PHYS_Falling)
	{
		bFalling = true;
	
		if(DoubleClickMove == DCLICK_Left)
			TraceEnd = vect(1, 0, 0);
		else if(DoubleClickMove == DCLICK_Right)
			TraceEnd = vect(-1, 0, 0);
		
		TraceStart = Location - (CylinderComponent.CollisionHeight - 16) * vect(0, 0, 1) + TraceEnd * (CylinderComponent.CollisionRadius - 16);
		TraceEnd = TraceStart + TraceEnd * 40;
		HitActor = Trace(HitLocation, HitNormal, TraceEnd, TraceStart, false, vect(16, 16, 16));
		
		HitNormal.Y = 0;
		HitNormal = normal(HitNormal);
		
		if((HitActor == none) || (HitNormal.Z < -0.1))
			return false;
		if(!HitActor.bWorldGeometry)
		{
			if(!HitActor.bBlockActors)
				return false;
			if((Pawn(HitActor) != none) && (Vehicle(HitActor) == none))
				return false;
		}
	}

	if(DoubleClickMove == DCLICK_Left)
	{
		Dir = vect(-1, 0, 0);
		Cross = vect(0, 1, 0);
	}
	else if(DoubleClickMove == DCLICK_Right)
	{
		Dir = vect(1, 0, 0);
		Cross = vect(0, 1, 0);
	}

	if(bFalling)
	{
		Velocity = vect(0, 0, 0);
		Acceleration = vect(0, 0, 0);
	}
	
	out = PerformDodge(DoubleClickMove, Dir, Cross);
	
	return out;
}

//Jump out of water always.
function bool CheckWaterJump(out vector WallNormal)
{
	return true;
}

//Block living things.
simulated function SetPawnRBChannels(bool bRagdollMode)
{
	super.SetPawnRBChannels(bRagdollMode);
	Mesh.SetRBCollidesWithChannel(RBCC_Untitled2, Health > 0);
}

state FeigningDeath
{	
	function AddVelocity(vector NewVelocity, vector HitLocation, class<DamageType> damageType, optional TraceHitInfo HitInfo);
}

//DefaultProperties
defaultproperties
{
	ControllerClass=class'UT2D.UT2DBot'
	bCanStrafe=false
	
	//Jumping
	JumpZ=550
	MultiJumpBoost=2
	OutofWaterZ=550
	MaxJumpHeight=132
	MaxDoubleJumpHeight=260
	
	SightRadius=500
}
