//-----------------------------------------------------------
// UT2DController_ConsolePlayer
//
// Copyright 2008 Joe Bates
//
// This file is part of UT2D.
//
// UT2D is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// UT2D is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with UT2D.  If not, see <http://www.gnu.org/licenses/>.
//
// Last Modified: October 28, 2008
//-----------------------------------------------------------
class UT2DController_ConsolePlayer extends UTConsolePlayerController;

//Camera Calculations
var vector LastViewLoc;
var vector CameraOffset;
var float LookingOffset;

//Aiming
var vector Look;
var float LookSensitivity, TurningThreshold;

//Simulating analog jump and crouch
var float AnalogJumpThreshold, AnalogDuckThreshold;
var bool bAnalogJumped;

//Ease slowly into crouch-looking
var float DuckTransitionTime;
var float BeganDucking;
var float StoppedDucking;

//Face the direction of newly spawned pawns
event Possess(Pawn inPawn, bool bVehicleTransition)
{
	Look = vector(inPawn.Rotation);
	super.Possess(inPawn, bVehicleTransition);
}

//Force third-person
function SetBehindView(bool bNewBehindView)
{
	super.SetBehindView(true);
}

//Third-person camera
simulated event GetPlayerViewPoint(out vector POVLocation, out Rotator POVRotation)
{
	local UT2DPawn P;
	local rotator rot;
	local vector vHeadOffset, vAimingOffset;
	local float scale;
	
	if(ViewTarget != none)
	{	
		P = UT2DPawn(ViewTarget);
		if(!IsInState('Dead') && !IsInState('RoundEnded') && P != none && P.IsLocallyControlled())
		{
			if(IsInState('Spectating'))
			{
				rot = P.Rotation;
				rot.Pitch = P.RemoteViewPitch << 8;
				Look = vector(rot);
			}
			
			vHeadOffset.Z = P.HeadOffset;
			LastViewLoc = P.Location + vHeadOffset;

			scale = 1;
			if(abs(Look.X) < TurningThreshold)
				scale += abs(Look.X) / TurningThreshold - 1;
			
			if(Look.X < 0)
				scale *= -1;
				
			vAimingOffset.X = LookingOffset * Scale;

			//Ducking
			if(BeganDucking != 0)
				scale = fmin(WorldInfo.TimeSeconds - BeganDucking, DuckTransitionTime) / DuckTransitionTime;
			else
                                scale = 1 - fmin(WorldInfo.TimeSeconds - StoppedDucking, DuckTransitionTime) / DuckTransitionTime;

			vAimingOffset.Z += P.EyeHeight - P.default.EyeHeight + LookingOffset * Look.Z * scale;

			LastViewLoc.Y = 0;
			POVLocation = LastViewLoc + vAimingOffset + CameraOffset;
		}
		else
		{
			if(!ViewTarget.IsA('UTGib'))
				LastViewLoc = ViewTarget.Location;

			if(ViewTarget == self)
			{
				if(LastViewLoc.Y == 0)
				{
					LastViewLoc += CameraOffset * 2;
					SetLocation(LastViewLoc);
				}

				bCollideWorld = true;
				POVLocation = LastViewLoc;
				
				LastViewLoc -= CameraOffset * 2;
			}
			else
			{
				LastViewLoc.Y = 0;
				POVLocation = LastViewLoc + CameraOffset;
			}
		}
	}
	else
		POVLocation = LastViewLoc + CameraOffset;
	
	POVRotation = rotator(-CameraOffset);
	
	StopViewShaking();
	
	if(CameraEffect != none)
		CameraEffect.UpdateLocation(POVLocation, POVRotation, GetFOVAngle());
}

//GetAdjustedAimFor
function rotator GetAdjustedAimFor(Weapon W, vector StartFireLoc)
{
	return Rotation;
}

//UpdateRotation
function UpdateRotation(float DeltaTime)
{
	local rotator DeltaRot;
	
	//Forget look inversion
	if(PlayerInput.bInvertMouse)
        	PlayerInput.aLookup = -PlayerInput.aLookup;
        if(PlayerInput.bInvertTurn)
        	PlayerInput.aTurn = -PlayerInput.aTurn;
	
	//Fake out UpdateRotation to rotate as our aiming system expects
	Look.Z += PlayerInput.aLookup * LookSensitivity;
	Look.X += PlayerInput.aTurn * LookSensitivity;
	Look = normal(Look);
	DeltaRot = rotator(Look) - Rotation;
	
	PlayerInput.aLookUp = DeltaRot.Pitch;
	PlayerInput.aTurn = DeltaRot.Yaw;
	
	super.UpdateRotation(DeltaTime);
}

//PlayerWalking
state PlayerWalking
{	
	//PlayerMove
	function PlayerMove(float DeltaTime)
	{
		local byte bOrigDuck;
		
		//Remember the original duck
		bOrigDuck = bDuck;
		
		//Analog up to jump
		if(PlayerInput.aBaseY > AnalogJumpThreshold)
		{
			if(!bAnalogJumped)
				bPressedJump = true;
			bAnalogJumped = true;
		}
		else
			bAnalogJumped = false;
		
		//Analog down to duck
		if(PlayerInput.aBaseY < AnalogDuckThreshold)
			bDuck = 1;

		if(Pawn.Physics == PHYS_Walking && BeganDucking == 0 && bDuck == 1)
		{
			BeganDucking = WorldInfo.TimeSeconds;
			if(BeganDucking - StoppedDucking < DuckTransitionTime)
				BeganDucking -= DuckTransitionTime - (BeganDucking - StoppedDucking);
		}
		else if(BeganDucking != 0 && (bDuck == 0 || Pawn.Physics != PHYS_Walking))
		{
			StoppedDucking = WorldInfo.TimeSeconds;
			if(StoppedDucking - BeganDucking < DuckTransitionTime)
				StoppedDucking -= DuckTransitionTime - (StoppedDucking - BeganDucking);

			BeganDucking = 0;
		}

		PlayerInput.aForward = 0;

		//Move based on side-scrolling
		Pawn.SetRotation(rot(0, -16384, 0));
		super.PlayerMove(DeltaTime);
		
		//Forget analog duck
		bDuck = bOrigDuck;
	}
}

//PlayerSwimming
state PlayerSwimming
{	
	//PlayerMove
	function PlayerMove(float DeltaTime)
	{	
		//Analog up to jump
		if(PlayerInput.aBaseY > AnalogJumpThreshold)
		{
			if(!bAnalogJumped)
				bPressedJump = true;
			bAnalogJumped = true;
		}
		else
			bAnalogJumped = false;
		
		//Move based on side-scrolling
		PlayerInput.aUp += PlayerInput.aForward;
		PlayerInput.aForward = 0;
		
		SetRotation(rot(0, -16384, 0));
		super.PlayerMove(DeltaTime);
		SetRotation(Pawn.Rotation);
	}
}

//PlayerClimbing
state PlayerClimbing
{	
	//PlayerMove
	function PlayerMove(float DeltaTime)
	{	
		//Move based on side-scrolling
		PlayerInput.aUp += PlayerInput.aForward;
		PlayerInput.aForward = 0;
		
		SetRotation(rot(0, -16384, 0));
		super.PlayerMove(DeltaTime);
		SetRotation(Pawn.Rotation);
	}
}

//PlayerFlying
state PlayerFlying
{	
	//PlayerMove
	function PlayerMove(float DeltaTime)
	{	
		//Move based on side-scrolling
		PlayerInput.aUp += PlayerInput.aForward;
		PlayerInput.aForward = 0;
		
		SetRotation(rot(0, -16384, 0));
		super.PlayerMove(DeltaTime);
		SetRotation(Pawn.Rotation);
	}
}

//PlayerWaiting
auto state PlayerWaiting
{	
	//PlayerMove
	function PlayerMove(float DeltaTime)
	{	
		//Move based on side-scrolling
		PlayerInput.aUp += PlayerInput.aForward;
		PlayerInput.aForward = 0;
		
		SetRotation(rot(0, -16384, 0));
		super.PlayerMove(DeltaTime);
	}
}

state WaitingForPawn
{
	//PlayerMove
	function PlayerMove(float DeltaTime)
	{	
		//Move based on side-scrolling
		PlayerInput.aUp += PlayerInput.aForward;
		PlayerInput.aForward = 0;
		
		SetRotation(rot(0, -16384, 0));
		super.PlayerMove(DeltaTime);
	}
}

state Dead
{
	//PlayerMove
	function PlayerMove(float DeltaTime)
	{	
		//Move based on side-scrolling
		PlayerInput.aUp += PlayerInput.aForward;
		PlayerInput.aForward = 0;
		
		SetRotation(rot(0, -16384, 0));
		super.PlayerMove(DeltaTime);
	}
}

state InQueue
{
	//PlayerMove
	function PlayerMove(float DeltaTime)
	{	
		//Move based on side-scrolling
		PlayerInput.aUp += PlayerInput.aForward;
		PlayerInput.aForward = 0;
		
		SetRotation(rot(0, -16384, 0));
		super.PlayerMove(DeltaTime);
	}
}

unreliable server function ServerViewSelf()
{	
	SetLocation(LastViewLoc + CameraOffset * 2);
	SetViewTarget(self);
}

state Spectating
{
	//PlayerMove
	function PlayerMove(float DeltaTime)
	{	
		//Move based on side-scrolling
		PlayerInput.aUp += PlayerInput.aForward;
		PlayerInput.aForward = 0;
		
		SetRotation(rot(0, -16384, 0));
		super.PlayerMove(DeltaTime);
	}
	
	exec function StartAltFire(optional byte FireModeNum)
	{
		ServerViewSelf();
	}
}

//DefaultProperties
defaultproperties
{	
	//Cheating
	CheatClass=class'UT2DCheatManager'
	
	//Camera
	bBehindView=true;
	CameraOffset=(X=0,Y=400,Z=32)
	LookingOffset=300
	
	//Aiming
	Look=(X=1,Y=0,Z=0)
	LookSensitivity=0.001
	TurningThreshold=0.85
	
	//Analog Simulation
	AnalogJumpThreshold=3000
	AnalogDuckThreshold=-3000
	
	//Ducking
	DuckTransitionTime=0.3
	BeganDucking=0
	StoppedDucking=-0.3
}