//============================================================
// TTeamFixMut.uc	- Class for managing mutator-specific events
//============================================================
//	TitanTeamFix
//		- A modular team balancing tool initially coded for the Titan servers:
//			http://ut2004.titaninternet.co.uk/
//
//	Copyright (C) 2007-2009 John "Shambler" Barrett (JBarrett847@Gmail.com or Shambler@OldUnreal.com)
//
//	This program is free software; you can redistribute and/or modify
//	it under the terms of the Open Unreal Mod License version 1.1.
//
//============================================================
Class TTeamFixMut extends Mutator;

const TTFVersion="v2.0beta5";
var deprecated bool bUpdateVersionNumber;


var TTeamFix Master;

// ===== Variables controlling used code
var bool bExitingEvent;
var bool bJoiningEvent;
var bool bJoinedEvent;
var bool bDelayedJoinedEvent;

var bool bChangingTeamEvent;
var bool bChangedTeamEvent;

var bool bAllowSpectateEvent;
var bool bPlayerToSpectatorEvent;

var bool bAllowSpectatorJoinEvent;
var bool bSpectatorToPlayerEvent;

var bool bMutateEvent;

var array<PlayerController> DelayedJoinNotify;

var class<TTeamFix> BalancerClass;


function PostBeginPlay()
{
	local Object TempObj;
	local UTUIDataProvider_Mutator DP;

	Super.PostBeginPlay();


	// Check that the data provider for this mutator exists (necessary for viewing the mutator by webadmin)
	TempObj = new(Class.Outer, "TitanTeamFix") Class'Package'; // I need to do this so that the data provider is written to the correct package
	DP = new(TempObj, string(Class)) Class'UTUIDataProvider_Mutator';

	if (DP.ClassName != PathName(Class))
	{
		LogInternal("Storing data provider for '"$Class$"'", 'TitanTeamFix');

		DP.ClassName = PathName(Class);
		Class'UnrealScriptTools2.USTools2'.static.SetFriendlyName(DP, "Titan Team Fix");
		Class'UnrealScriptTools2.USTools2'.static.SetDescription(DP, "Configurable Team Balancer");

		DP.SaveConfig();
	}

	if (bDelayedJoinedEvent && !WorldInfo.Game.bKickMissingCDHashKeys)
	{
		LogInternal("Forcing bKickMissingCDHashKeys to True, TTF needs this setting", 'TitanTeamFix');

		WorldInfo.Game.bKickMissingCDHashKeys = True;
		WorldInfo.Game.SaveConfig();
	}
}


// Used to re-initialize mutator after seamless travelling
function InitMutator(string Options, out string ErrorMessage)
{
	local TTeamFix TTF;

	Super.InitMutator(Options, ErrorMessage);

	if (Master == none)
	{
		foreach AllActors(Class'TTeamFix', TTF)
		{
			if (UTTeamGame(WorldInfo.Game) != none && TTF.EventMutator != Self)
			{
				if (TTF.Class == BalancerClass)
					TTF.InitializeBalancing();
				else if (!TTF.bInitializedBalancing)
					TTF.bBlockInitialization = True;
			}

			break;
		}
	}


	// TTF needs to run as a mutator now due to the UT3 WebAdmin not allowing modifications to the commandline; if you make a new
	// balancer class, you will also need to make a new mutator class to go with it (with a modified 'BalancerClass' variable)
	// (this may change in the future, if serveractors are fixed in a patch)
	if (Master == none)
	{
		TTF = Spawn(BalancerClass);
		TTF.InitializeBalancing();
	}
}


function InitializeEvents(TTeamFix Caller)
{
	// Some checking statements
	if (Caller == none)
	{
		// Just added to remind me to update version numbers each time I recompile
		bUpdateVersionNumber = True;

		WarnInternal("TTeamFixMut::InitializeEvents Caller is none");
		return;
	}

	if (Master != none && Master != Caller)
		WarnInternal("TTeamFixMut::InitializeEvents Master was already set, initializing from new Caller and disregarding old Master");

	Master = Caller;

	bExitingEvent = Master.bNeedExitingEvent;
	bJoiningEvent = Master.bNeedJoiningEvent;
	bJoinedEvent = Master.bNeedJoinedEvent;
	bDelayedJoinedEvent = Master.bNeedDelayedJoinedEvent;
	bChangingTeamEvent = Master.bNeedChangingTeamEvent;
	bChangedTeamEvent = Master.bNeedChangedTeamEvent;
	bAllowSpectateEvent = Master.bNeedBecomingSpectatorEvent;
	bPlayerToSpectatorEvent = Master.bNeedBecameSpectatorEvent;
	bAllowSpectatorJoinEvent = Master.bNeedSpectatorBecomingPlayerEvent;
	bSpectatorToPlayerEvent = Master.bNeedSpectatorBecamePlayerEvent;
	bMutateEvent = Master.bNeedMutateEvent;
}

function NotifyLogout(Controller Exiting)
{
	Super.NotifyLogout(Exiting);

	if (bExitingEvent && Master != none && Exiting != none)
	{
		// When players are in the middle of exiting the game, they are still counted on their old team; this accounts for that.
		// N.B. I'm using a dynamic array as a gross-overcompensation for the unlikely event of NotifyLogout being called recursively
		Master.ExitingPlayers.AddItem(Exiting);

		Master.PlayerExitingGame(Exiting);

		Master.ExitingPlayers.Length = Master.ExitingPlayers.Length - 1;
	}
}

function ModifyLogin(out string Portal, out string Options)
{
	if (bJoiningEvent && Master != none)
		Master.PlayerJoiningGame(Portal, Options);

	Super.ModifyLogin(Portal, Options);
}

function NotifyLogin(Controller NewPlayer)
{
	Super.NotifyLogin(NewPlayer);

	if (Master == none)
		return;


	if (bJoinedEvent)
		Master.PlayerJoinedGame(NewPlayer);


	if (bDelayedJoinedEvent)
	{
		if (PlayerController(NewPlayer) != none && (PlayerController(NewPlayer).HashResponseCache == "" || PlayerController(NewPlayer).HashResponseCache == "0"))
		{
			DelayedJoinNotify.AddItem(PlayerController(NewPlayer));
			Enable('Tick');
		}
		else
		{
			Master.DelayedJoinedGame(NewPlayer);
		}
	}
}

// Broken with patch 2.0, fixed in 2.1
function bool AllowChangeTeam(Controller Other, out int num, bool bNewTeam)
{
	local bool bResult;

	bResult = True;

	if (bChangingTeamEvent && Master != none)
		bResult = Master.PlayerChangingTeam(Other, num, bNewTeam);

	// Allow other mutators to block a teamchange, but not allow one if the team balancer has blocked it
	if (bResult && NextMutator != none)
		bResult = NextMutator.AllowChangeTeam(Other, num, bNewTeam);


	return bResult;
}

function NotifySetTeam(Controller Other, TeamInfo OldTeam, TeamInfo NewTeam, bool bNewTeam)
{
	Super.NotifySetTeam(Other, OldTeam, NewTeam, bNewTeam);

	if (bChangedTeamEvent && Master != none)
		Master.PlayerChangedTeam(Other, OldTeam, NewTeam, bNewTeam);
}

function bool AllowBecomeSpectator(PlayerController P)
{
	local bool bResult;

	bResult = True;

	if (bAllowSpectateEvent && Master != none)
		bResult = Master.PlayerBecomingSpectator(P);

	// Allow other mutators to block a player from becoming a spectator, but not allow it if the team balancer blocked it
	if (bResult && NextMutator != none)
		bResult = NextMutator.AllowBecomeSpectator(P);


	return bResult;
}

function NotifyBecomeSpectator(PlayerController Player)
{
	Super.NotifyBecomeSpectator(Player);

	if (bPlayerToSpectatorEvent && Master != none)
		Master.PlayerBecameSpectator(Player);
}

function bool AllowBecomeActivePlayer(PlayerController P)
{
	local bool bResult;

	bResult = True;

	if (bAllowSpectatorJoinEvent && Master != none)
		bResult = Master.SpectatorBecomingPlayer(P);

	// Allow other mutators to block the spectator from joining, but not allow it if the team balancer blocked it
	if (bResult && NextMutator != none)
		bResult = NextMutator.AllowBecomeActivePlayer(P);


	return bResult;
}

function NotifyBecomeActivePlayer(PlayerController Player)
{
	Super.NotifyBecomeActivePlayer(Player);

	if (bSpectatorToPlayerEvent && Master != none)
		Master.SpectatorBecamePlayer(Player);
}

function Tick(float DeltaTime)
{
	local int i;

	for (i=0; i<DelayedJoinNotify.Length; ++i)
	{
		if (DelayedJoinNotify[i] == none)
		{
			DelayedJoinNotify.Remove(i--, 1);
			continue;
		}

		if (DelayedJoinNotify[i].HashResponseCache != "" && DelayedJoinNotify[i].HashResponseCache != "0")
		{
			Master.DelayedJoinedGame(DelayedJoinNotify[i]);
			DelayedJoinNotify.Remove(i--, 1);
		}
	}


	if (DelayedJoinNotify.Length <= 0)
		Disable('Tick');
}

function GetSeamlessTravelActorList(bool bToEntry, out array<Actor> ActorList)
{
	Super.GetSeamlessTravelActorList(bToEntry, ActorList);

	if (Master != none)
		Master.NotifySeamlessTravel(bToEntry, ActorList);
}


function Mutate(string MutateString, PlayerController Sender)
{
	if (Master == none || Sender == none)
	{
		Super.Mutate(MutateString, Sender);
		return;
	}


	if (MutateString ~= "TTFVersion")
	{
		Sender.ClientMessage("Running TitanTeamFix version:"@TTFVersion);
	}
	else if (!bMutateEvent || Master.NotifyMutate(MutateString, Sender))
	{
		Super.Mutate(MutateString, Sender);
	}
}


defaultproperties
{
	BalancerClass=Class'TTeamFixGeneric'
}