//============================================================
// TTeamFixGeneric.uc	- Gametype independant team balancing class, the actual balancer is implemented here
//============================================================
//	TitanTeamFix
//		- A modular team balancing tool initially coded for the Titan servers:
//			http://ut2004.titaninternet.co.uk/
//
//	Copyright (C) 2007-2009 John "Shambler" Barrett (JBarrett847@Gmail.com or Shambler@OldUnreal.com)
//
//	This program is free software; you can redistribute and/or modify
//	it under the terms of the Open Unreal Mod License version 1.1.
//
//============================================================
//
// Highly configurable team balancer class.
// It's possible to tweak this to suit a variety of different
// gametypes, all without touching any code.
//
//============================================================
Class TTeamFixGeneric extends TTeamFix;


// ===== Variables set by the config profile

// The course of action to take when the teams have become uneven
enum EImbalanceAction
{
	IA_NoAction,			// For when you want no automatic balancing
	IA_Countdown,			// Counts down from 'BalanceCountdown' seconds and if the teams are still uneven, it forcibly rebalances the teams
	IA_DeathEvent,			// Monitors players that have been ingame for less than two minutes, and switches any that die until the teams balance
	IA_CountdownOrDeath,		// A combination of IE_Countdown and IE_DeathEvent
	IA_TeamDeath			// Monitors for the death of an entire team and then forcibly rebalances the teams (for round based gametypes)
};

enum EPreferredTeams
{
	PT_Disable,
	PT_Enable,
	PT_PreferLosingTeam
};

enum EShuffleTeams
{
	ST_Disable,
	ST_MatchStart,
	ST_MatchEnd
};

enum EShuffleMode
{
	SM_Random,
	SM_Points,
	SM_PointsVsDeaths,
	SM_PointsPerSecond,
	SM_DamageDealt,
	SM_DamageDealtVsDamageTaken,
	SM_DamageDealtPerSecond,
	SM_ELORank
};

struct PlayerDamageInfo
{
	var Controller C;
	var int DamageDealt;
	var int DamageTaken;
};


var EImbalanceAction ImbalanceAction;	// The action to undertake when the teams become uneven
var int BalanceCountdown;		// If ImbalanceAction is set to 'IA_Countdown' or 'IA_CountdownOrDeath' then this is the countdown time

// Variables related to messages
var bool bAnnounceSwitch;		// Tells all the players that 'blah' has been switched
var bool bAnnounceImbalance;		// When the teams become imbalanced, notify all players
var bool bAnnounceSlotOpened;		// When a server is full and an exiting player opens a slot, notify all spectators

var EPreferredTeams PreferredTeams;	// Preferred team settings, allows the server to disable preferred teams or set players to join the losing team
var EShuffleTeams ShuffleTeams;		// Shuffe teams settings, allows the server to shuffle the teams at the start of a match or at the end
var EShuffleMode ShuffleMode;		// What method to use in order to shuffle the teams

var bool bBotsBalanceTeams;		// If true, bots are used to keep the teams balanced
var int BotBalanceCutoff;		// If 'bBotsBalanceTeams', then bots are no longer used to balance the teams when the playercount exceeds this number

// =====

var bool bPendingBalance;		// If true, the auto-balance countdown is active and/or the code is waiting for a recently joined player to die and switch
var int MinPlayers;			// If 'bBotsBalanceTeams', then this is set to match the gameinfo's DesiredPlayerCount

var Controller PendingSwapDeath;	// The player currently changing team, who might trigger an unwanted 'PlayerKilled' event
var float LastPendingSwapDeath;		// 'PendingSwapDeath' is only valid during the same tick it was set; this timestamp is used to validate it

var array<PlayerDamageInfo> DamageTracker;	// When shuffling is enabled, with ShuffleMode set to SM_DamageDealt(PerSecond), damage tracking is performed here

// Variables used as compiler messages
var deprecated bool bTODO;


// ===== General functions
function InitializeBalancing()
{
	Super.InitializeBalancing();

	if (bBotsBalanceTeams && !UTGame(WorldInfo.Game).bPlayersVsBots)
	{
		// Temporarily disable bBotsBalanceTeams, since this is called before DesiredPlayerCount is set and must be delayed by a tick
		bBotsBalanceTeams = False;
		SetTimer(0.001, False, 'InitializeMinPlayers');
	}
}

// Called first-tick after InitGame, so that I can grab UTGame.DesiredPlayerCount
function InitializeMinPlayers()
{
	MinPlayers = UTGame(WorldInfo.Game).DesiredPlayerCount;


	// Upscale MinPlayers if it isn't an even number
	MinPlayers += MinPlayers % 2;

	// Downscale MinPlayers if it exceeds MaxPlayers
	if (MinPlayers > WorldInfo.Game.MaxPlayers)
		MinPlayers -= 2;


	UTGame(WorldInfo.Game).DesiredPlayerCount = (BotBalanceCutoff == 0 ? MinPlayers : BotBalanceCutoff);


	bBotsBalanceTeams = True;
	ClearTimer('InitializeMinPlayers');
}

function InitializeDefaults()
{
	local Mutator M;


	// ***
	bTODO = True;
	// Remove all of the commented code, after testing is complete
	// ***


	// Lots of bug reports with BattleTeamArena; want to make sure it's not due to bad configuration
	if (ImbalanceAction != IA_NoAction && ImbalanceAction != IA_TeamDeath && ImbalanceAction != IA_DeathEvent
		&& ConfigObject.Name == 'default' &&
		(string(WorldInfo.Game.Class) ~= "BattleTeamArena" || string(WorldInfo.Game.Class) ~= "BattleFreezetagArena" ||
		string(WorldInfo.Game.Class) ~= "BattleInstagibArena" || string(WorldInfo.Game.Class) ~= "BattleInstaFreezeArena"))
	{
		`log("BattleTeamArena is running with the 'default' configuration profile, forcing ImbalanceAction to IA_TeamDeath",, 'TitanTeamFix');
		`log("- NOTE: To override this, use a config profile other than 'default'",, 'TitanTeamFix');

		ImbalanceAction = IA_TeamDeath;
	}


	if (ShuffleTeams != ST_Disable)
	{
		for (M=WorldInfo.Game.BaseMutator; M!=none; M=M.NextMutator)
		{
			if (string(M.Class) ~= "GoWJustice")
			{
				`log("The 'GoW Justice' mutator is running, disabling TitanTeamFix team shuffling, to prevent conflicts",, 'TitanTeamFix');
				ShuffleTeams = ST_Disable;

				break;
			}
		}

		if (ShuffleTeams != ST_Disable && (ShuffleMode == SM_DamageDealt || ShuffleMode == SM_DamageDealtVsDamageTaken ||
			ShuffleMode == SM_DamageDealtPerSecond))
		{
			bNeedDamageEvent = True;
		}
	}


	if (ImbalanceAction == IA_DeathEvent || ImbalanceAction == IA_CountdownOrDeath || ImbalanceAction == IA_TeamDeath)
		bNeedKilledEvent = True;


	if (PreferredTeams == PT_Disable || PreferredTeams == PT_PreferLosingTeam)
		bNeedJoiningEvent = True;

	if (bBotsBalanceTeams)
	{
		bNeedJoinedEvent = True;
		bNeedSpectatorBecamePlayerEvent = True;
	}

	// Always need exiting event, since v2.0
	bNeedExitingEvent = True;
	bNeedBecameSpectatorEvent = True;

	if (ImbalanceAction != IA_NoAction || bAnnounceImbalance)
		bNeedChangedTeamEvent = True;


	// *****
	bTODO = True;
	// TODO: Remove this debug code! (when removing it though, check that you haven't started using it for something useful)
	bNeedMutateEvent = True;
	// *****
}


// Implements team-shuffling on server travel
function ResetBalancing()
{
	Super.ResetBalancing();

	// Either shuffle the teams now if set to execute at matchend, or if score/damage based shuffling is enabled, store the data for the next level
	if (ShuffleTeams == ST_MatchEnd)
		RandomizeTeams();
	else if (ShuffleTeams != ST_Disable && ShuffleMode != SM_Random)
		Class'TTeamFixPersistentData'.static.GetPersistentDataObj().StorePlayerData(Self);
}

function MatchStarting()
{
	local int BiggerTeam, Imbalance;

	// Implements team-shuffling on server travel
	if (ShuffleTeams == ST_MatchStart)
	{
		RandomizeTeams();
	}
	else if (ImbalanceAction != IA_NoAction)
	{
		// If balancing is enabled (and 'shuffle teams' disabled), check that the teams are balanced at match start
		if (bTeamsUneven(BiggerTeam, Imbalance))
			EvenTeams(BiggerTeam, Imbalance, True);
	}
}

// TODO: Clean up and optimize this function (also rename it, as name is no longer accurate; many of the vars are badly named too, and aren't reused enough)
function RandomizeTeams()
{
	local PlayerController PC;
	local int i, j, BiggerTeam, Imbalance, CurRank;
	local array<PlayerController> RebalanceList, RankedList;
	local int CantSwitchCount[2];
	local int NewTeams[2];
	local int RankedDistribution[2];
	local array<int> TeamAssignments, BiggerTeamIndicies, RankedValues;
	local TTeamFixPersistentData PersistentObj;
	local AIController AC;
	local array<AIController> BotList;
	local bool OldbBotsBalanceTeams, bReverseRank;
	local int OldDesiredPlayerCount;

	// TODO: Remove this commented code
	/*
	foreach WorldInfo.AllControllers(Class'PlayerController', PC)
	{
		if (PC.PlayerReplicationInfo != none && !PC.PlayerReplicationInfo.bOnlySpectator && Rand(2) != PC.GetTeamNum())
		{
			TTFChangeTeam(PC, int(!bool(PC.GetTeamNum())));
		}
	}


	// If the teams were unbalanced through randomization, fix that
	if (bTeamsUneven(BiggerTeam, Imbalance,, True))
	{
		foreach WorldInfo.AllControllers(Class'PlayerController', PC)
			if (PC.GetTeamNum() == BiggerTeam && PC.PlayerReplicationInfo != none && !PC.PlayerReplicationInfo.bOnlySpectator)
				RebalanceList.InsertItem(0, PC);

		// Decide upon the number of players to switch, (Imbalance & 0xFFFE) cuts odd numbers, turning 3 into 2, 5 into 4 etc.
		RebalanceList.Length = (Imbalance & 0xFFFE) / 2;
		BiggerTeam = int(!bool(BiggerTeam));

		foreach RebalanceList(PC)
			TTFChangeTeam(PC, BiggerTeam);
	}
	*/

	// Gather the player list and count the players who can't be switched
	foreach WorldInfo.AllControllers(Class'PlayerController', PC)
	{
		if (PC.PlayerReplicationInfo != none && !PC.PlayerReplicationInfo.bOnlySpectator)
		{
			if (PC.HasClientLoadedCurrentWorld())
				RebalanceList.AddItem(PC);
			else
				++CantSwitchCount[PC.GetTeamNum()];
		}
	}

	if (RebalanceList.Length != 0)
	{
		// Determine the type of team shuffling to use, and rebalance the lists appropriately

		// Score/damage based shuffling (randomly re-assigns players later on, if 'RebalanceList' isn't emptied)
		if (ShuffleMode != SM_Random)
		{
			PersistentObj = Class'TTeamFixPersistentData'.static.GetPersistentDataObj();

			// NOTE: The persistent data storage object is used here even when it's not needed, primarily for code-consistency
			// ***
			bTODO = True;
			// Tweak the code so that the persistent data object is only used when needed
			// ***
			if (ShuffleTeams == ST_MatchStart)
				PersistentObj.SetupPlayerData(Self);
			else
				PersistentObj.StorePlayerData(Self, True);


			for (i=0; i<PersistentObj.StoredPlayerData.Length; ++i)
			{
				bReverseRank = False;

				switch (ShuffleMode)
				{
					case SM_Points:
						CurRank = Max(PersistentObj.StoredPlayerData[i].Score, 0.0);
						break;

					case SM_PointsVsDeaths:
						CurRank = int((Max(float(PersistentObj.StoredPlayerData[i].Score), 0.0)/
								(Max(float(PersistentObj.StoredPlayerData[i].Deaths), 0.0)+1.0)) * 100.0);
						break;

					case SM_PointsPerSecond:
						CurRank = int((Max(float(PersistentObj.StoredPlayerData[i].Score), 0.0)/
								float(PersistentObj.StoredPlayerData[i].PlayTime)) * 10000.0);
						break;

					case SM_DamageDealt:
						CurRank = PersistentObj.StoredPlayerData[i].DamageDealt;
						break;

					case SM_DamageDealtVsDamageTaken:
						CurRank = int((float(PersistentObj.StoredPlayerData[i].DamageDealt)/
								(Max(float(PersistentObj.StoredPlayerData[i].DamageTaken), 1.0))) * 100.0);
						break;

					case SM_DamageDealtPerSecond:
						CurRank = int((float(PersistentObj.StoredPlayerData[i].DamageDealt)/
								float(PersistentObj.StoredPlayerData[i].PlayTime)) * 10000.0);
						break;

					case SM_ELORank:
						CurRank = PersistentObj.StoredPlayerData[i].ELORank;
						break;
				}

				for (j=0; j<RankedValues.Length; ++j)
				{
					if ((!bReverseRank && RankedValues[j] <= CurRank) || RankedValues[j] > CurRank)
					{
						RankedValues.Insert(j, 1);
						RankedList.Insert(j, 1);

						break;
					}
				}

				if (j == RankedValues.Length)
				{
					RankedValues.Length = j+1;
					RankedList.Length = j+1;
				}

				RankedValues[j] = CurRank;
				RankedList[j] = PersistentObj.AssociatedControllers[i];
			}


			// Now that 'RankedList' is sorted, switch all players present in 'RebalanceList', and remove them as you go
			CurRank = Rand(2);

			// ***
			bTODO = True;
			// Remove this debug log (after online testing is complete)
			`log("TTeamFixGeneric::RandomizeTeams: ShuffleTeams:"@GetEnum(Enum'EShuffleTeams', ShuffleTeams)$", ShuffleMode:"@
				GetEnum(Enum'EShuffleMode', ShuffleMode),, 'TTFDebug');
			// ***

			for (i=0; i<RankedList.Length; ++i)
			{
				j = RebalanceList.Find(RankedList[i]);

				if (j != INDEX_None)
				{
					RebalanceList.Remove(j, 1);


					// If putting the player on the other team would lead to more balanced teams, without messing up the numbers, do so
					if (NewTeams[CurRank] > NewTeams[Abs(CurRank-1)] && RankedDistribution[Abs(CurRank-1)] <= RankedDistribution[CurRank])
						CurRank = Abs(CurRank-1);


					// ***
					bTODO = True;
					// Remove this debug code (AFTER IT HAS BEEN TESTED ONLINE)
					`log("TTeamFixGeneric::RandomizeTeams: Rank:"@i$", RankScore:"@RankedValues[i]$", TeamAssignment:"@CurRank
						$", PlayerName:"@RankedList[i].PlayerReplicationInfo.PlayerName,, 'TTFDebug');
					// ***


					++CantSwitchCount[CurRank];
					++RankedDistribution[CurRank];
					NewTeams[CurRank] += RankedValues[i];

					TTFChangeTeam(RankedList[i], CurRank);
					CurRank = Abs(CurRank-1);
				}
			}


			// Clean up the persistent data, stored from the previous level
			PersistentObj.ClearPlayerData();
		}


		// Randomly assign team numbers
		if (RebalanceList.Length > 0 || ShuffleMode == SM_Random)
		{
			TeamAssignments.Length = RebalanceList.Length;
			NewTeams[0] = 0;
			NewTeams[1] = 0;

			for (i=0; i<TeamAssignments.Length; ++i)
			{
				TeamAssignments[i] = Rand(2);
				++NewTeams[TeamAssignments[i]];
			}


			// Check if the assigned team numbers will unbalance the teams, and if so, correct the values by randomly switching back players
			Imbalance = Abs((NewTeams[0] + CantSwitchCount[0]) - (NewTeams[1] + CantSwitchCount[1]));

			if (Imbalance >= 2)
			{
				BiggerTeam = int((NewTeams[1] + CantSwitchCount[1]) > (NewTeams[0] + CantSwitchCount[0]));

				for (i=0; i<TeamAssignments.Length; ++i)
					if (TeamAssignments[i] == BiggerTeam)
						BiggerTeamIndicies.AddItem(i);

				while (Imbalance >= 2 && BiggerTeamIndicies.Length > 0)
				{
					i = Rand(BiggerTeamIndicies.Length);

					TeamAssignments[BiggerTeamIndicies[i]] = Abs(BiggerTeam-1);

					--NewTeams[BiggerTeam];
					++NewTeams[TeamAssignments[i]];

					Imbalance -= 2;
					BiggerTeamIndicies.Remove(i, 1);
				}
			}
		}


		// Now switch the players
		for (i=0; i<RebalanceList.Length; ++i)
		{
			// ***
			bTODO = True;
			// Remove this debug code (AFTER IT HAS BEEN TESTED ONLINE)
			if (ShuffleMode != SM_Random)
			{
				`log("TTeamFixGeneric::RandomizeTeams: Randomly reassigning unranked players, Index:"@i$", TeamAssignment:"@TeamAssignments[i]$
					", PlayerName:"@RebalanceList[i].PlayerReplicationInfo.PlayerName,, 'TTFDebug');
			}
			else
			{
				`log("TTeamFixGeneric::RandomizeTeams: Randomly shuffling players, Index:"@i$", TeamAssignment:"@TeamAssignments[i]$
					", PlayerName:"@RebalanceList[i].PlayerReplicationInfo.PlayerName,, 'TTFDebug');
			}
			// ***

			TTFChangeTeam(RebalanceList[i], TeamAssignments[i]);
		}
	}


	// If there are any bots present, then the reshuffling code may have imbalanced the teams due to ignoring the bots.
	// To fix this, destroy the bots; UT should automatically re-add them as they get destroyed, distributing them so as to rebalance the teams
	if (ShuffleTeams != ST_MatchEnd || !UTTeamGame(WorldInfo.Game).bPlayersBalanceTeams)
	{
		// Temporarily disable 'bBotsBalanceTeams' and 'DesiredPlayerCount', so that UT does not immediately re-add the bots
		OldbBotsBalanceTeams = bBotsBalanceTeams;
		OldDesiredPlayerCount = UTGame(WorldInfo.Game).DesiredPlayerCount;
		bBotsBalanceTeams = False;
		UTGame(WorldInfo.Game).DesiredPlayerCount = 0;


		foreach WorldInfo.AllControllers(Class'AIController', AC)
			BotList.AddItem(AC);


		// I assign bots to 'BotList' before destroying them, because destroying them within the above iterator may result in an infinite loop,
		// as UT will automatically add new bots as I destroy the old ones
		foreach BotList(AC)
		{
			if (AC.Pawn != none)
				AC.Pawn.Destroy();

			if (AC != none)
				AC.Destroy();
		}


		// Restore values
		bBotsBalanceTeams = OldbBotsBalanceTeams;
		UTGame(WorldInfo.Game).DesiredPlayerCount = OldDesiredPlayerCount;
	}


	// Notify all players that the teams have been randomized
	if (bAnnounceSwitch)
		foreach WorldInfo.AllControllers(Class'PlayerController', PC)
			PC.ReceiveLocalizedMessage(Class'TTFMsg.TTeamFixMessages', 5);
}


// Adjusts incoming players preferred team
function PlayerJoiningGame(out string Portal, out string Options)
{
	local int ScoreDiff, i, j;
	local string NewOpt;

	// Adjust the players preferred team, if appropriate
	if (PreferredTeams == PT_PreferLosingTeam)
	{
		if (GRI == none)
			GRI = WorldInfo.Game.GameReplicationInfo;

		// First determine the losing team
		ScoreDiff = GRI.Teams[0].Score - GRI.Teams[1].Score;

		if (ScoreDiff < 0)
			ScoreDiff = 0;
		else if (ScoreDiff > 0)
			ScoreDiff = 1;
		else
			ScoreDiff = -1;


		// Apply the selected team to the preffered team parameter
		i = InStr(Caps(Options), "?TEAM=");
		NewOpt = Mid(Options, i+1);
		j = InStr(NewOpt, "?");

		if (ScoreDiff != -1)
		{
			if (j != -1)
				NewOpt = Left(Options, i)$"?Team="$ScoreDiff$Mid(NewOpt, j);
			else
				NewOpt = Left(Options, i)$"?Team="$ScoreDiff;
		}
		// NOTE: Modified in beta4, as sometimes a player would be placed to stack teams
		else
		{
			if (j != -1)
				NewOpt = Left(Options, i)$Mid(NewOpt, j);
			else
				NewOpt = Left(Options, i);
		}

		Options = NewOpt;
	}
	else if (PreferredTeams == PT_Disable)
	{
		i = InStr(Caps(Options), "?TEAM=");
		NewOpt = Mid(Options, i+1);
		j = InStr(NewOpt, "?");

		if (j != -1)
			NewOpt = Left(Options, i)$Mid(NewOpt, j);
		else
			NewOpt = Left(Options, i);

		Options = NewOpt;
	}
}

// Currently used to help make bots balance the teams
function PlayerJoinedGame(Controller Player)
{
	if (bBotsBalanceTeams && PlayerController(Player) != none && (WorldInfo.Game.GetNumPlayers() % 2) != 0)
	{
		UTGame(WorldInfo.Game).DesiredPlayerCount = Min(Max(MinPlayers, WorldInfo.Game.GetNumPlayers() + 1),
								(BotBalanceCutoff == 0 ? WorldInfo.Game.MaxPlayers : BotBalanceCutoff));
	}
}

function SpectatorBecamePlayer(PlayerController Player)
{
	// ***
	bTODO = True;
	// Check that this fixes the problem in your notes, relating to bots balancing teams not working correctly when a spectator joins
	// ***

	// ***
	bTODO = True;
	// It's unclear whether or not the player will be affecting 'GetNumPlayers()' at this time, and the same in 'PlayerJoinedGame';
	//	you will need to check this and make sure that no bugs are caused by this
	// ***

	// ***
	bTODO = True;
	// After you have checked the above two 'TODO' notes in this function, move the bot balance code to its own function
	// ***

	PlayerJoinedGame(Player);
}

// N.B. Remember that this gets called for exiting players BEFORE the team sizes have changed
function PlayerKilled(Controller Player)
{
	local int BiggerTeam, Imbalance;
	local PlayerController CurPC;
	local array<PlayerController> PC;

	// Ignore calls triggered from a team-swap death
	if (LastPendingSwapDeath == WorldInfo.TimeSeconds && PendingSwapDeath == Player)
		return;


	if (bPendingBalance && bTeamsUneven(BiggerTeam, Imbalance))
	{
		if (ImbalanceAction == IA_TeamDeath)
		{
			if (CheckTeamDeath(Player))
				EvenTeams(BiggerTeam, Imbalance);
		}
		else if (ImbalanceAction == IA_DeathEvent || ImbalanceAction == IA_CountdownOrDeath)
		{
			CurPC = PlayerController(Player);

			// Keep this if statement seperate in case I decide to add in more death related events
			if (CurPC != none && CurPC.GetTeamNum() == BiggerTeam && bRecentlyJoined(CurPC) && bCanSwitchPlayer(CurPC))
			{
				PC.AddItem(CurPC);
				NotifySwitchPlayers(PC);

				SwitchPlayer(CurPC);

				if (!bTeamsUneven(BiggerTeam, Imbalance))
					SetPendingBalance(False);
			}
		}
	}
}

// Treat players becoming spectators as players exiting
function PlayerBecameSpectator(PlayerController Player)
{
	// ***
	bTODO = True;
	// Test this with that spectate mutator, to make sure that the player becoming a spectator, is not still registered as a player
	// ***

	PlayerExitingGame(Player);
}

function PlayerExitingGame(Controller Player)
{
	local PlayerController PC;
	local int i;

	if (bBotsBalanceTeams)
	{
		if ((WorldInfo.Game.GetNumPlayers() % 2) != 0)
		{
			UTGame(WorldInfo.Game).DesiredPlayerCount = Min(Max(MinPlayers, WorldInfo.Game.GetNumPlayers() + 1),
									(BotBalanceCutoff == 0 ? WorldInfo.Game.MaxPlayers : BotBalanceCutoff));
		}
		else
		{
			UTGame(WorldInfo.Game).DesiredPlayerCount = Min(Max(MinPlayers, WorldInfo.Game.GetNumPlayers()),
									(BotBalanceCutoff == 0 ? WorldInfo.Game.MaxPlayers : BotBalanceCutoff));
		}
	}

	if (DamageTracker.Length > 0)
	{
		i = DamageTracker.Find('C', Player);

		if (i != INDEX_None)
			DamageTracker.Remove(i, 1);
	}

	// NOTE: You don't need to add IA_TeamDeath checks here, because exiting players trigger 'PlayerKilled' before this
	CheckForImbalance();


	// If the server was full, and the exiting player opens up a player slot, then notify all spectators that a slot has been opened
	if (bAnnounceSlotOpened && PlayerController(Player) != none && !Player.PlayerReplicationInfo.bOnlySpectator
		&& WorldInfo.Game.GetNumPlayers() + 1 == WorldInfo.Game.MaxPlayers)
	{
		foreach WorldInfo.AllControllers(Class'PlayerController', PC)
			if (PC.PlayerReplicationInfo != none && PC.PlayerReplicationInfo.bOnlySpectator)
				PC.ReceiveLocalizedMessage(Class'TTFMsg.TTeamFixMessages', 4);
	}
}

function PlayerChangedTeam(Controller Player, TeamInfo OldTeam, TeamInfo NewTeam, bool bNewTeam)
{
	PendingSwapDeath = Player;
	LastPendingSwapDeath = WorldInfo.TimeSeconds;

	CheckForImbalance();
}

function NotifyPlayerDamaged(int Damage, pawn Injured, Controller InstigatedBy)
{
	local int i;

	if (Injured.Controller != none && (Injured.DrivenVehicle == none || InstigatedBy == none || InstigatedBy.Pawn != Injured.DrivenVehicle))
	{
		// Entry for the attacking player
		if (PlayerController(InstigatedBy) != none && InstigatedBy != Injured.Controller && Injured.GetTeamNum() != InstigatedBy.GetTeamNum())
		{
			i = DamageTracker.Find('C', InstigatedBy);

			if (i == INDEX_None)
			{
				i = DamageTracker.Length;
				DamageTracker.Length = i+1;

				DamageTracker[i].C = InstigatedBy;
			}

			DamageTracker[i].DamageDealt += Damage;
		}


		// Entry for the hurt player
		if (PlayerController(Injured.Controller) != none &&
			(InstigatedBy == none || Injured.GetTeamNum() != InstigatedBy.GetTeamNum() || Injured.Controller == InstigatedBy))
		{
			i = DamageTracker.Find('C', Injured.Controller);

			if (i == INDEX_None)
			{
				i = DamageTracker.Length;
				DamageTracker.Length = i+1;

				DamageTracker[i].C = Injured.Controller;
			}

			DamageTracker[i].DamageTaken += Damage;
		}
	}
}


final function CheckForImbalance()
{
	local int BiggerTeam, Imbalance, IdleCount;

	if (bTeamsUneven(BiggerTeam, Imbalance, IdleCount))
	{
		if (!bPendingBalance)
			NotifyImbalance(IdleCount);
	}
	else if (bPendingBalance)
	{
		SetPendingBalance(False);
	}
}

// Only notifies when bPendingBalance was false
function NotifyImbalance(optional int IdleCount)
{
	local PlayerController PC;

	if (bAnnounceImbalance)
	{
		foreach WorldInfo.AllControllers(Class'PlayerController', PC)
		{
			PC.ReceiveLocalizedMessage(Class'TTFMsg.TTeamFixMessages', (IdleCount<<3) + 3);
		}
	}


	switch (ImbalanceAction)
	{
	case IA_Countdown:
	case IA_CountdownOrDeath:
		SetPendingBalance(True, True);
		break;

	case IA_DeathEvent:
	case IA_TeamDeath:
		SetPendingBalance(True);
		break;

	default:
	}
}


function SetPendingBalance(bool bPending, optional bool bEnableCountdown)
{
	bPendingBalance = bPending;

	if (bPending)
	{
		if (bEnableCountdown && !IsTimerActive('CountdownTimer'))
		{
			if (BalanceCountdown > 0)
				SetTimer(BalanceCountdown,, 'CountdownTimer');
			else
				CountdownTimer();
		}
	}
	else
	{
		ClearTimer('CountdownTimer');
	}
}

function CountdownTimer()
{
	local int BiggerTeam, Imbalance;

	if (bTeamsUneven(BiggerTeam, Imbalance))
		EvenTeams(BiggerTeam, Imbalance);

	SetPendingBalance(False);
}

function NotifySwitchPlayers(array<PlayerController> Players)
{
	local PlayerController PC;
	local int Indicies;

	if (bAnnounceSwitch)
	{
		// Notify the players that are being switched
		foreach Players(PC)
			PC.ReceiveLocalizedMessage(Class'TTFMsg.TTeamFixMessages', 0);


		// Make a general announcement to all other players
		if (Players.Length == 1)
		{
			foreach WorldInfo.AllControllers(Class'PlayerController', PC)
				if (Players.Find(PC) == -1)
					PC.ReceiveLocalizedMessage(Class'TTFMsg.TTeamFixMessages', 1, Players[0].PlayerReplicationInfo);
		}
		else
		{
			Indicies = -1;

			// N.B. Also sent to players that are being switched; so that they can see who else is being switched too
			foreach WorldInfo.AllControllers(Class'PlayerController', PC)
			{
				if (Indicies == -1)
					Indicies = EncodeMultipleAnnounce(Players);

				PC.ReceiveLocalizedMessage(Class'TTFMsg.TTeamFixMessages', Indicies, Players[0].PlayerReplicationInfo,,
								WorldInfo.Game.GameReplicationInfo);
			}
		}
	}
}

// Special function for squeezing in extra data to the 'Switch' parameter in ReceiveLocalizedMessage
final function int EncodeMultipleAnnounce(array<PlayerController> Players)
{
	local int CurTeam, i, ReturnVal;
	local PlayerReplicationInfo PRI, PRI2;
	local array<PlayerReplicationInfo> PRIList;

	// ReturnVal represents the final 'Switch' value; which must be 2 for MultipleAnnounce messages
	ReturnVal = 2;

	CurTeam = Players[0].GetTeamNum();

	// Generate the list of players from the bigger team and sort by player id
	foreach WorldInfo.Game.GameReplicationInfo.PRIArray(PRI)
	{
		if (PRI.GetTeamNum() != CurTeam)
			continue;

		if (PRIList.Length == 0)
		{
			PRIList.AddItem(PRI);
		}
		else
		{
			foreach PRIList(PRI2, i)
			{
				if (PRI.PlayerID < PRI2.PlayerID)
				{
					PRIList.InsertItem(i, PRI);
					break;
				}
			}

			if (PRIList[i] != PRI)
				PRIList.AddItem(PRI);
		}
	}

	// Now compare entries in the list to the 'Players' array and note them in the return value
	foreach PRIList(PRI, i)
		if (PlayerController(PRI.Owner) != none && Players.Find(PlayerController(PRI.Owner)) != -1)
			ReturnVal = ReturnVal | (0x8 << i);


	return ReturnVal;
}


defaultproperties
{
	ConfigProfileClass=Class'TTFProfile_Generic'
}