//============================================================
// TTeamFixConfigProfile.uc	- An object for storing multiple configuration profiles
//============================================================
//	TitanTeamFix
//		- A modular team balancing tool initially coded for the Titan servers:
//			http://ut2004.titaninternet.co.uk/
//
//	Copyright (C) 2007-2009 John "Shambler" Barrett (JBarrett847@Gmail.com or Shambler@OldUnreal.com)
//
//	This program is free software; you can redistribute and/or modify
//	it under the terms of the Open Unreal Mod License version 1.1.
//
//============================================================
//
// This class stores configuration profiles and allows admins
// to specify a seperate name for each profile.
// It also allows console commands and other code to display
// detailed documentation about properties and their values
//
// N.B. This profile class is only a stub for other profile
// configuration classes
//
// N.B. Properties that should be configurable through the
// WebAdmin, must be marked as databinding, and have
// appropriate 'Properties'/'PropertyMappings'/
// 'PropertyDescriptions' setup in the default properties
// (check TTFProfile_Generic.uc for examples)
//
//============================================================
class TTeamFixConfigProfile extends Settings
	PerObjectConfig
	config(TitanTeamFix);

var array<string> PropertyDescriptions;

var class<TTeamFix> BalancerClass;
var string WebAdminHookClass;	// The class used to initialize this objects web admin hook (must be a subclass of 'TTeamFixWebAdminHook')


final function TransferProperties(TTeamFix Owner)
{
	if (Class != Owner.ConfigProfileClass)
	{
		WarnInternal("Config profile attached to invalid owner");
		return;
	}

	ValidateProperties(Owner);
	CopyFromProfile(Owner);
}

final function SaveProperties(TTeamFix Owner)
{
	if (Owner.Class != BalancerClass)
	{
		WarnInternal("Config profile attached to invalid owner");
		return;
	}

	CopyToProfile(Owner);
	ValidateProperties(Owner);

	SaveConfig();
}


// Implement in subclasses
function CopyFromProfile(TTeamFix Owner);
function CopyToProfile(TTeamFix Owner);
function ValidateProperties(optional TTeamFix Owner); // For fixing up properties that depend upon each other


function InitWebAdminProperties(TTeamFix Owner)
{
	local array<Name> DBProperties;
	local int i;
	local string CurProperty;

	// IMPORTANT: Many complex property types are not supported by this function (e.g. object/struct properties)
	GetDataBindingProperties(DBProperties);

	for (i=0; i<DBProperties.Length; ++i)
		if (GetDataBindingValueSafe(DBProperties[i], CurProperty))
			SetPropertyFromStringByName(DBProperties[i], CurProperty);
}

function SaveWebAdminProperties(TTeamFix Owner, out string Warning, out string Error)
{
	local array<Name> DBProperties;
	local int i;
	local string CurProperty;

	GetDataBindingProperties(DBProperties);

	for (i=0; i<DBProperties.Length; ++i)
	{
		CurProperty = GetPropertyAsStringByName(DBProperties[i]);
		SetDataBindingValue(DBProperties[i], CurProperty);
	}

	ValidateProperties(Owner);
	InitWebAdminProperties(Owner); // In case 'ValidateProperties' changed some
	SaveConfig();
}

function bool GetDataBindingValueSafe(name PropertyName, out string Value)
{
	local Property PropertyObj;
	local string PropertyValue;

	// First get a reference to the property object, to help determine its type
	PropertyObj = Property(FindObject(PathName(Class)$":"$PropertyName, Class'Property'));

	if (PropertyObj == none)
		return False;


	// Now get the raw string value for the property
	if (!GetDataBindingValue(PropertyName, PropertyValue))
		return False;


	// Now determine how the value should be converted

	// If it's an enum property, then get it's raw numeric value
	if (IsDataBindingEnum(PropertyName, PropertyValue))
	{
		Value = string(DataBindingEnumToByte(ByteProperty(PropertyObj), PropertyValue));
		return True;
	}

	// If it's a bool property, return the raw number value instead of true/false
	if (BoolProperty(PropertyObj) != none)
	{
		Value = string(byte(bool(PropertyValue)));
		return True;
	}

	Value = PropertyValue;
	return True;
}

// Converts the string value of a databinding enum property, to a byte value
final function byte DataBindingEnumToByte(ByteProperty EnumProperty, const out string EnumValue)
{
	local byte i;
	local string TestValue;

	for (i=0; i<255; ++i)
		if (SetDataBindingValue(EnumProperty.Name, string(i)) && GetDataBindingValue(EnumProperty.Name, TestValue) && TestValue == EnumValue)
			break;


	// Return the property to its original value
	SetDataBindingValue(EnumProperty.Name, EnumValue);

	return i;
}

// Returns true if the specific databinding property is an enum
final function bool IsDataBindingEnum(name PropertyName, optional out string OrigValue)
{
	local ByteProperty PropertyObj;
	local string TestValue;
	local int ValueLen, i, j;

	// First get a reference to the property object, to help determine its type
	PropertyObj = ByteProperty(FindObject(PathName(Class)$":"$PropertyName, Class'Property'));

	if (PropertyObj == none || (OrigValue == "" && !GetDataBindingValue(PropertyName, OrigValue)))
		return False;


	// Set the value to 0 (i.e. the first enum value), in case it's out of range of the enum
	SetDataBindingValue(PropertyName, "0");
	GetDataBindingValue(PropertyName, TestValue);
	SetDataBindingValue(PropertyName, OrigValue);

	ValueLen = Len(TestValue);

	// First verify that the value is an enum value
	for (i=0; i<ValueLen; ++i)
	{
		j = Asc(Mid(TestValue, i, 1));

		// Check if the value is not a number
		if (j < 48 || j > 57)
			return True;
	}


	return False;
}


defaultproperties
{
	BalancerClass=Class'TTeamFix'
}