//============================================================
// TTeamFix.uc	- This class will be the placeholder for the balancing system
//============================================================
//	TitanTeamFix
//		- A modular team balancing tool initially coded for the Titan servers:
//			http://ut2004.titaninternet.co.uk/
//
//	Copyright (C) 2007-2009 John "Shambler" Barrett (JBarrett847@Gmail.com or Shambler@OldUnreal.com)
//
//	This program is free software; you can redistribute and/or modify
//	it under the terms of the Open Unreal Mod License version 1.1.
//
//============================================================
//
// This class is designed to be a modular base class for all team-balancing components,
// it is designed around certain key events like for a player joining a team or leaving
// the game etc. and these key events are called by either the 'EventMutator' or the
// 'EventRules' objects.
//
// The reason I want this to be modular is so that anybody can code plugins for this
// mutator so it can run better on different gametypes.
//
//============================================================
Class TTeamFix extends Info;


// The variables that affect which players are chosen for switching
enum ESwitchVariable
{
	SV_SwitchCount,
	SV_JoinTime,		// TODO: Should there be a time margin between players (say 10-20 seconds) where they fit the same 'time slot'?
	SV_Score,
	SV_Deaths,
	SV_Health,
	SV_KillingSpree,
	SV_HasSuperWeapon,
	SV_HasSuperPickup,
	SV_InVehicle,
	SV_AliveTime,
	SV_Admin,
	SV_FlagDistance		// Includes flag carrier distance, both friendly and enemy, and flagbase/node distance (only enemy/untaken)
};

// Whether or not a specified switch variable puts a player higher or lower in the list
enum ESwitchPlacement
{
	SP_Low,
	SP_High
};

struct SwitchOrderElement
{
	var ESwitchVariable	Variable;
	var ESwitchPlacement	Placement;
};

// *****
// TODO: Remove this commented code
/*
struct SwitchGroup
{
	var array<PlayerController>	Players;
	var array<SwitchGroup>		SubGroups;
	var int				GroupDepth;
};
*/
// *****

struct SwitchGroup
{
	var int StartIdx;	// Index into a dynamic array of playercontrollers
	var int EndIdx;		// ^^
};

// *****
// TODO: Remove this debug code
// Struct to aid debugging code
struct DebugSwitchGroup extends SwitchGroup
{
	var array<int>			SubGroups;	// Index into the next 'DebugSwitchGroupTable' column
	var SwitchOrderElement		SwitchInfo;
	var bool			bPostEarlyExit;
};

struct DebugSwitchGroupTable
{
	var array<DebugSwitchGroup>	Columns;
};
// *****

// Change these values in subclasses via the default properties, in most cases the original defaults will suit your purposes
var class<TTeamFixMut>			EventMutClass;
var class<TTeamFixRules>		EventRulesClass;
var class<TTeamFixConfigProfile>	ConfigProfileClass;		// The config profile class which corresponds to this teamfix class


// Events that require 'EventRules' (set these in InitializeDefaults)
var bool				bNeedPreSpawnEvent;
var bool				bNeedKilledEvent;
var bool				bNeedRestartGameEvent;
var bool				bNeedDamageEvent;

// Events that require 'EventMutator'
var bool				bNeedExitingEvent;
var bool				bNeedJoiningEvent;
var bool				bNeedJoinedEvent;
var bool				bNeedDelayedJoinedEvent;	// A version of the 'PlayerJoinedGame' event which is delayed for 10 seconds or until the players ID has been retrieved
var bool				bNeedChangingTeamEvent;
var bool				bNeedChangedTeamEvent;
var bool				bNeedBecomingSpectatorEvent;
var bool				bNeedBecameSpectatorEvent;
var bool				bNeedSpectatorBecomingPlayerEvent;
var bool				bNeedSpectatorBecamePlayerEvent;
var bool				bNeedMutateEvent;

// Used as config variables in TTeamFixGeneric
var bool				bIgnoreIdlePlayers;		// If true, then TTF doesn't count idle players when determining if the teams are uneven
var int					IgnoreIdleTime;			// The amount of time a player can be inactive before TTF considers him/her to be idle
var array<SwitchOrderElement>		SwitchOrder;			// The order in which players are chosen for switching (in order of precedence)


// ===== Runtime variables
var TTeamFixMut				EventMutator;
var TTeamFixRules			EventRules;
var TTeamFixConfigProfile		ConfigObject;			// Object which contains the config profile for this class (NOTE: This can be none, e.g. if the webadmin deletes it)
var bool				bInitializedBalancing;
var bool				bBlockInitialization;		// Block initializing of the team balancer, so another can be created instead
var bool				bSetupWebAdminHook;
var Class<TTeamFixWebAdminHook>		WebAdminHookClass;

var GameReplicationInfo			GRI;
var array<Controller> 			ExitingPlayers;
var int					RecentCutoffTime;

struct SwitchList
{
	var PlayerController Player;
	var byte SwitchCount;
};

var array<SwitchList>			SwitchHistory;

// *****
// TODO: Remove this debug code
var array<DebugSwitchGroupTable>	LastSwitchGroupTree;		// Used to let admins dump the most recent switch group tree
var array<PlayerController>		LastSwitchGroupPlayers;
var float				LastSwitchGroupTimestamp;
var float				LastWriteTimestamp;
var string				LastWriteFilename;		// TODO: Make sure you set this
// *****


// Variables used as compiler messages
var deprecated bool bCheckAgainstPatch_LastVer_Patch4;
var deprecated bool bTODO;



// ===== Functions relying on outside-class events from 'EventMutator' and 'EventRules'

// 'EventRules' events
function PlayerSpawning(controller Player, out byte InTeam);
singular function PlayerKilled(controller Player);
function NotifyRestartGame();
function NotifyPlayerDamaged(int Damage, pawn Injured, Controller InstigatedBy);


// 'EventMutator' events

// N.B. 'PlayerKilled' is called just before this when the player is exiting
function PlayerExitingGame(Controller Player);

function PlayerJoiningGame(out string Portal, out string Options);

function PlayerJoinedGame(Controller Player/*, optional bool bInvalidId*/);

// N.B. When bNeedDelayedJoinedEvent == True 'PlayerSpawning' is called BEFORE this when a player joins
function DelayedJoinedGame(Controller Player, optional bool bInvalidId);


// Called when a player ATTEMPTS to change team; return false to prevent a teamchange
// NOTE: The mutator hook for this got clobbered before 2.0 release; broken (use 'PlayerChangedTeam' to put players back instead), fixed in 2.1
function bool PlayerChangingTeam(Controller Player, out int num, bool bNewTeam)
{
	return True;
}

// Called after a player successfully changes team
function PlayerChangedTeam(Controller Player, TeamInfo OldTeam, TeamInfo NewTeam, bool bNewTeam);


// Called when a player ATTEMPTS to become a spectator; return false to prevent this
function bool PlayerBecomingSpectator(PlayerController Player)
{
	return True;
}

// Called after a player successfully becomes a spectator
function PlayerBecameSpectator(PlayerController Player);

// Called when a spectator ATTEMPTS to become an active player; return false to prevent this
function bool SpectatorBecomingPlayer(PlayerController Player)
{
	return True;
}

// Called after a spectator successfully becomes a player
function SpectatorBecamePlayer(PlayerController Player);

// Return true if the command has been handled here, and you don't want to continue passing it along
function bool NotifyMutate(string Command, PlayerController Sender)
{
	local TTeamFixDebugFileWriter DFW;
	local int i, j, k;
	local array<int> CurBranch;
	local bool bStart;

	Command = Class'UTUIScene'.static.TrimWhitespace(Command);

	if (Command ~= "TTFDumpList")
	{
		if (!Sender.PlayerReplicationInfo.bAdmin)
		{
			Command = Class'UTUIScene'.static.TrimWhitespace(Mid(Command, 12));

			// Allow non-admins to call the command, if they know the pass; it probably can't be abused anyway, but
			//	I figure it's best to at least add a pass just in case. This way I can still test it myself without
			//	admin, if need be
			if (!(Left(Command, 8) == "NotAdmin"))
			{
				Sender.ClientMessage("TitanTeamFix: You need admin access to use this command");
				return True;
			}
		}


		if (LastSwitchGroupTimestamp == 0.0 || LastSwitchGroupTree.Length == 0)
		{
			Sender.ClientMessage("TitanTeamFix: No switch list available for dumping");
			return True;
		}
		else if (LastSwitchGroupTimestamp == LastWriteTimestamp)
		{
			Sender.ClientMessage("TitanTeamFix: The current switch list has already been dumped, filename:"@LastWriteFilename);
			return True;
		}


		// Ready to dump the switch tree list, setup the file writer and begin
		DFW = Class'TTeamFixDebugFileWriter'.static.GetDebugWriter(Self, "SwitchTree");

		// Base loop
		for (i=0; i<LastSwitchGroupTree.Length; ++i)
		{
			// Keep at start
			k = CurBranch.AddItem(i);

			// Log the element info
			DebugDumpSwitchTreeElement(DFW, LastSwitchGroupTree[i].Columns[k]);


			// Branch loop
			bStart = True;
			j = 0;

			while (k > 0 || (bStart && LastSwitchGroupTree[i].Columns[k].SubGroups.Length > 0))
			{
				bStart = False;

			ExtendBranch:
				while (j<LastSwitchGroupTree[i].Columns[k].SubGroups.Length)
				{
					// Keep at start
					k = CurBranch.AddItem(LastSwitchGroupTree[i].Columns[k].SubGroups[j]);
					DFW.IncreaseIndent();

					// Log the sub-element info
					DebugDumpSwitchTreeElement(DFW, LastSwitchGroupTree[i].Columns[k]);

					goto 'ExtendBranch';
				}

				// Keep at end
				j = LastSwitchGroupTree[i].Columns[k-1].SubGroups.Find(CurBranch[k]) + 1;
				CurBranch.Length = k--;
				DFW.DecreaseIndent();
			}


			// Keep at end
			CurBranch.Length = k--;
		}

		Sender.ClientMessage("TitanTeamFix: Dumped switch list to file '"$DFW.Filename$"'");
		LastWriteFilename = DFW.Filename;
		LastWriteTimestamp = LastSwitchGroupTimestamp;
		DFW.Close();

		return True;
	}

	return False;
}


function NotifySeamlessTravel(bool bToEntry, out array<Actor> ActorList)
{
	// NOTE: This shouldn't be needed
	//ActorList[ActorList.Length] = Self;

	// ResetBalancing gives you the chance to unload any per-level objects, if necessary
	if (bToEntry)
		ResetBalancing();
}


// 'TTeamFix' events (called from this class)

// This particular 'event' is nearly useless, I've only added it so that the switch-announcement code can be implemented in a subclass
function NotifySwitchPlayers(array<PlayerController> Players);


// ===== Debug functions
final function DebugDumpSwitchTreeElement(TTeamFixDebugFileWriter DFW, DebugSwitchGroup CurElement, optional bool bForceDumpPlayers)
{
	local int i, Val;
	local array<actor> DesiredObjectives;

	bTODO = True;
	// TODO: REMOVE THIS DEBUG FUNCTION WHEN DONE WITH IT!!


	// Element info
	DFW.Log("-"@GetEnum(Enum'ESwitchVariable', CurElement.SwitchInfo.Variable)@
		GetEnum(Enum'ESwitchPlacement', CurElement.SwitchInfo.Placement)@"(StartIdx:"@CurElement.StartIdx$", EndIdx:"@
		CurElement.EndIdx$", bPostEarlyExit:"@CurElement.bPostEarlyExit$")");

	// Player info (if necessary)
	if (bForceDumpPlayers || CurElement.SubGroups.Length == 0)
	{
		DFW.IncreaseIndent();

		if (CurElement.SwitchInfo.Variable == SV_FlagDistance && CurElement.StartIdx != CurElement.EndIdx)
			GetTeamObjectiveList(DesiredObjectives, LastSwitchGroupPlayers[CurElement.StartIdx].GetTeamNum());

		for (i=CurElement.StartIdx; i<CurElement.EndIdx; ++i)
		{
			DFW.Log("- Player:"@LastSwitchGroupPlayers[i].PlayerReplicationInfo.PlayerName);

			// Increase indent and log the active variable for this player
			DFW.IncreaseIndent();

			switch(CurElement.SwitchInfo.Variable)
			{
				case SV_SwitchCount:
					Val = SwitchHistory.Find('Player', LastSwitchGroupPlayers[i]);

					if (Val == -1)
						Val = 0;
					else
						Val = SwitchHistory[Val].SwitchCount;

					break;

				case SV_JoinTime:
					Val = LastSwitchGroupPlayers[i].PlayerReplicationInfo.StartTime;
					break;

				case SV_Score:
					Val = LastSwitchGroupPlayers[i].PlayerReplicationInfo.Score;
					break;

				case SV_Deaths:
					Val = LastSwitchGroupPlayers[i].PlayerReplicationInfo.Deaths;
					break;

				case SV_Health:
					Val = ((!LastSwitchGroupPlayers[i].IsDead() && LastSwitchGroupPlayers[i].Pawn != none) ? (LastSwitchGroupPlayers[i].Pawn.Health +
						UTPawn(LastSwitchGroupPlayers[i].Pawn).GetShieldStrength()) : 0);
					break;

				case SV_KillingSpree:
					Val = UTPlayerReplicationInfo(LastSwitchGroupPlayers[i].PlayerReplicationInfo).Spree;
					break;

				case SV_HasSuperWeapon:
					Val = int(bHasSuperPickup(LastSwitchGroupPlayers[i], True));
					break;

				case SV_HasSuperPickup:
					Val = int(bHasSuperPickup(LastSwitchGroupPlayers[i]));
					break;

				case SV_InVehicle:
					Val = int(UTVehicle(LastSwitchGroupPlayers[i].Pawn) != none && UTVehicle_Hoverboard(LastSwitchGroupPlayers[i].Pawn) == none);
					break;

				case SV_AliveTime:
					Val = (LastSwitchGroupPlayers[i].Pawn != none ? int(WorldInfo.TimeSeconds - LastSwitchGroupPlayers[i].Pawn.SpawnTime) : 0);
					break;

				case SV_Admin:
					Val = int(LastSwitchGroupPlayers[i].PlayerReplicationInfo.bAdmin);
					break;

				case SV_FlagDistance:
					Val = NearestObjectiveDist(LastSwitchGroupPlayers[i], DesiredObjectives);
					break;
			}

			DFW.Log("-"@GetEnum(Enum'ESwitchVariable', CurElement.SwitchInfo.Variable)@"value:"@Val@
					"(NOTE: Player values taken at time of mutate command, and may not represent values at time of balancing)");

			DFW.DecreaseIndent();
		}

		DFW.DecreaseIndent();
	}
}


// ===== General functions

// Use this only in classes which should be loaded from 'ServerActors'
/*
function PostBeginPlay()
{
	if (UTTeamGame(WorldInfo.Game) != none && WorldInfo.GetMapName() != "EnvyEntry")
		InitializeBalancing();
}
*/

function InitializeBalancing()
{
	local Mutator m;
	local GameRules gr;
	local TTeamFixMessageReplicationInfo TTFMRI;

	if (bInitializedBalancing)
		return;


	bInitializedBalancing = True;

	// Load the assigned config profile and setup the property defaults
	LoadPresetConfig();
	InitializeDefaults();


	// NOTE: Not always reliable, this function also gets called from InitGame->InitMutator, which is before the GRI is spawned
	GRI = WorldInfo.Game.GameReplicationInfo;


	// Make sure the EventMutator class is not already in the mutator list
	if (EventMutator == none)
		for (m=WorldInfo.Game.BaseMutator; m!=none; m=m.NextMutator)
			if (TTeamFixMut(m) != none)
				EventMutator = TTeamFixMut(m);

	// Same with rules list
	if (EventRules == none)
		for (gr=WorldInfo.Game.GameRulesModifiers; gr!=none; gr=gr.NextGameRules)
			if (TTeamFixRules(gr) != none)
				EventRules = TTeamFixRules(gr);


	if (NeedMutator())
	{
		// Initiate the mutator and gamerules objects
		if (EventMutator == none)
		{
			EventMutator = Spawn(EventMutClass);
			EventMutator.bUserAdded = True;

			if (WorldInfo.Game.BaseMutator == none)
				WorldInfo.Game.BaseMutator = EventMutator;
			else
				WorldInfo.Game.BaseMutator.AddMutator(EventMutator);
		}

		EventMutator.InitializeEvents(Self);
	}


	if (NeedRules())
	{
		if (EventRules == none)
		{
			EventRules = Spawn(EventRulesClass);

			if (WorldInfo.Game.GameRulesModifiers == none)
			{
				WorldInfo.Game.GameRulesModifiers = EventRules;
			}
			else if (bNeedDamageEvent)
			{
				// Add to the start of the list for the damage hook, in case other mods/muts modify damage there
				EventRules.NextGameRules = WorldInfo.Game.GameRulesModifiers;
				WorldInfo.Game.GameRulesModifiers = EventRules;
			}
			else
			{
				WorldInfo.Game.GameRulesModifiers.AddGameRules(EventRules);
			}
		}

		EventRules.InitializeEvents(Self);
	}


	// If the server uses custom TTF messages, setup the replication info
	if (Class'TTeamFixMessageReplicationInfo'.default.bEnableCustomMessages)
	{
		foreach AllActors(Class'TTeamFixMessageReplicationInfo', TTFMRI)
			continue;

		if (TTFMRI == none)
		{
			`log("Creating replication info for custom messages",, 'TitanTeamFix');
			Spawn(Class'TTeamFixMessageReplicationInfo');
		}
	}
}

function InitializeWebAdminHook()
{
	local object WebServerObj;

	if (WebAdminHookClass != none)
		return;

	WebServerObj = UTGame(WorldInfo.Game).WebServerObject;

	if (WebServerObj != none)
	{
		WebAdminHookClass = Class<TTeamFixWebAdminHook>(DynamicLoadObject(ConfigProfileClass.default.WebAdminHookClass, Class'Class'));

		if (WebAdminHookClass != none)
			WebAdminHookClass.static.InitializeWebAdminHook(Self, WebServerObj);
	}
}

// Implement in subclasses
function ResetBalancing();
function InitializeDefaults();


// Configuration profile loading
function LoadPresetConfig()
{
	local string ConfigName, SuffixConfigName, FinalConfig;
	local array<string> KnownProfiles;
	local int sLen, cLen, i;

	if (ConfigProfileClass == none)
		return;


	// Set the timer for initializing the web admin hook; it can't be initialized now as the web admin wont start until the next tick
	if (Class'TTeamFixConfigLoader'.default.bAddWebAdminConfigMenu && !bSetupWebAdminHook && ConfigProfileClass.default.WebAdminHookClass != "")
	{
		// This 'should' call the timer during the next tick
		SetTimer(-1.0, False, 'InitializeWebAdminHook');
		bSetupWebAdminHook = True;
	}

	// Make sure the config loader has an entry in UTGame.ini, so that admins can change config profiles
	Class'TTeamFixConfigLoader'.static.StaticSaveConfig();

	// Use the TTeamFixConfigLoader class (which stores config info in UTGame.ini) to find the desired config set
	ConfigName = Class'TTeamFixConfigLoader'.default.ActiveConfigurationProfile;

	if (Class'TTeamFixConfigLoader'.default.bSuffixGameToProfile)
	{
		SuffixConfigName = string(WorldInfo.Game.Class);
		SuffixConfigName = ConfigName$"_"$SuffixConfigName;
	}


	// Check that the config profile exists (n.b. output values will be in the format "ConfigName ConfigProfileClass")
	if (GetPerObjectConfigSections(ConfigProfileClass, KnownProfiles))
	{
		// The code doesn't appear to like string@string within condition statements
		SuffixConfigName @= "";
		ConfigName @= "";
		sLen = Len(SuffixConfigName);
		cLen = Len(ConfigName);

		for (i=0; i<KnownProfiles.Length; ++i)
		{
			// If the suffixed profile exists, it has precedence over the non-suffixed profile...store it and break
			if (SuffixConfigName != "" && (Left(KnownProfiles[i], sLen) ~= SuffixConfigName))
			{
				FinalConfig = Left(SuffixConfigName, sLen-1);
				break;
			}

			if (Left(KnownProfiles[i], cLen) ~= ConfigName)
				FinalConfig = Left(ConfigName, cLen-1);
		}

		SuffixConfigName = Left(SuffixConfigName, sLen-1);
		ConfigName = Left(ConfigName, cLen-1);
	}

	// Log success/failure in finding configuration profiles
	if (SuffixConfigName != "" && FinalConfig != SuffixConfigName)
		LogInternal("Could not find configuration profile '"$SuffixConfigName$"' for loading", 'TitanTeamFix');
	else if (ConfigName != "" && FinalConfig != ConfigName && FinalConfig != SuffixConfigName)
		LogInternal("Could not find configuration profile '"$ConfigName$"' for loading", 'TitanTeamFix');


	// Assign the found configuration, defaulting if not found
	if (FinalConfig == "")
		ConfigName = "default";
	else
		ConfigName = FinalConfig;


	LogInternal("Loading configuration profile '"$ConfigName$"'", 'TitanTeamFix');


	// Load the object which contains the specified config information
	//ConfigObject = new(none, ConfigName) ConfigProfileClass;
	ConfigObject = Class'TTeamFixConfigLoader'.static.FindConfigProfile(ConfigName, ConfigProfileClass);

	// If the default config profile was loaded, make sure that profile exists (otherwise, create it)
	if (ConfigName ~= "default")
	{
		for (i=0; i<KnownProfiles.Length-1; ++i)
			if (Left(KnownProfiles[i], 8) ~= "default ")
				break;

		if (i >= KnownProfiles.Length || !(Left(KnownProfiles[i], 8) ~= "default "))
		{
			LogInternal("Default profile did not exist, creating default profile", 'TitanTeamFix');

			ConfigObject.SaveConfig();
		}
	}

	// Transfer the properties
	ConfigObject.TransferProperties(Self);
}


// ===== Checking functions

// This returns true if the current configuration needs events from the gamerules class
function bool NeedRules()
{
	return (bNeedPreSpawnEvent || bNeedKilledEvent || bNeedRestartGameEvent);
}

function bool NeedMutator()
{
	return (bNeedExitingEvent || bNeedJoiningEvent || bNeedJoinedEvent || bNeedDelayedJoinedEvent || bNeedChangingTeamEvent || bNeedChangedTeamEvent
		|| bNeedBecomingSpectatorEvent || bNeedBecameSpectatorEvent || bNeedSpectatorBecomingPlayerEvent || bNeedSpectatorBecamePlayerEvent);
}

function bool bTeamsUneven(optional out int BiggerTeam, optional out int Imbalance, optional out int IdleCount, optional bool bForceCheck)
{
	local int TeamCount[2];
	local PlayerController PC;


	if (GRI == none)
	{
		GRI = WorldInfo.Game.GameReplicationInfo;

		if (GRI == none)
			return false;
	}


	if (!bForceCheck && WorldInfo.Game.bGameEnded)
		return false;


	foreach WorldInfo.AllControllers(Class'PlayerController', PC)
	{
		if (ExitingPlayers.Find(PC) == -1 && PC.PlayerReplicationInfo != none && !PC.PlayerReplicationInfo.bOnlySpectator &&
			PC.HasClientLoadedCurrentWorld())
		{
			if (!bIgnoreIdlePlayers || bForceCheck || WorldInfo.TimeSeconds - PC.LastActiveTime < IgnoreIdleTime)
				++TeamCount[PC.GetTeamNum()];
			else
				++IdleCount;
		}
	}


	Imbalance = Abs(TeamCount[0] - TeamCount[1]);
	BiggerTeam = int(TeamCount[1] > TeamCount[0]);

	// The teams are only considered to be uneven if a team has 2 or more extra players
	if (Imbalance >= 2)
		return True;

	return false;
}

final function bool bRecentlyJoined(Controller C)
{
	if (GRI == none)
	{
		GRI = WorldInfo.Game.GameReplicationInfo;

		if (GRI == none)
			return True;
	}

	return (GRI.ElapsedTime - C.PlayerReplicationInfo.StartTime <= RecentCutoffTime);
}

final function bool bHasSuperPickup(Controller C, optional bool bWeapon)
{
	local Pawn MainPawn;
	local UTWeapon W;

	if (Vehicle(C.Pawn) != none)
		MainPawn = Vehicle(C.Pawn).Driver;
	else
		MainPawn = C.Pawn;

	if (MainPawn == none || MainPawn.InvManager == none)
		return False;


	if (bWeapon)
	{
		foreach MainPawn.InvManager.InventoryActors(Class'UTWeapon', W)
		{
			if (W.bSuperWeapon && W.HasAmmo(0))
				return True;
		}
	}
	else
	{
		return MainPawn.InvManager.FindInventoryType(Class'UTTimedPowerup', True) != none;
	}

	return False;
}

final function GetTeamObjectiveList(out array<Actor> ObjectiveList, int Team)
{
	local UTGameObjective Obj;
	local int i;

	foreach WorldInfo.AllNavigationPoints(Class'UTGameObjective', Obj)
	{
		if (Obj.IsNeutral() || Obj.GetTeamNum() == int(!bool(Team)))
			ObjectiveList.AddItem(Obj);
	}

	if (UTOnslaughtGame(WorldInfo.Game) != none)
	{
		for (i=0; i<2; ++i)
		{
			if (UTOnslaughtGame(WorldInfo.Game).Orbs[i] != none && !UTOnslaughtGame(WorldInfo.Game).Orbs[i].bHome &&
				((i == Team && !UTOnslaughtGame(WorldInfo.Game).Orbs[Team].IsNearlyHome()) ||
					UTOnslaughtGame(WorldInfo.Game).Orbs[Team].IsInState('Dropped')))
			{
				ObjectiveList.AddItem(UTOnslaughtGame(WorldInfo.Game).Orbs[i]);
			}
		}
	}

	if (UTCTFGame(WorldInfo.Game) != none)
	{
		for (i=0; i<2; ++i)
		{
			if (UTCTFGame(WorldInfo.Game).Flags[i] != none && (i != Team ||
				(i == Team && UTCTFGame(WorldInfo.Game).Flags[i].IsInState('Dropped'))))
			{
				ObjectiveList.AddItem(UTCTFGame(WorldInfo.Game).Flags[i]);
			}
		}
	}
};

// Returns -1 for an error
final function int NearestObjectiveDist(Controller C, const out array<Actor> DesiredObjectives)
{
	local Pawn P;
	local int i, CurVal, BestVal;

	P = C.Pawn;

	if (P == none)
		return -1;


	BestVal = -1;

	for (i=0; i<DesiredObjectives.Length; ++i)
	{
		CurVal = VSize(DesiredObjectives[i].Location - P.Location);

		if (i == 0 || CurVal < BestVal)
			BestVal = CurVal;
	}

	return BestVal;
}

// Returns true if all the players on either team are dead
// TODO: Perhaps modify the 'PlayerKilled' event so that you don't have to specify the 'NewDeath' parameter when calling this from that function
function bool CheckTeamDeath(optional Controller NewDeath)
{
	local Controller C;
	local byte TeamIsAlive[2];

	foreach WorldInfo.AllControllers(Class'Controller', C)
	{
		if (C == NewDeath || ExitingPlayers.Find(C) != -1)
			continue;

		if (!C.IsDead() && !C.PlayerReplicationInfo.bOutOfLives && C.GetTeamNum() < 2)
			TeamIsAlive[C.GetTeamNum()] = 1;
	}

	return !bool(TeamIsAlive[0]) || !bool(TeamIsAlive[1]);
}

function EvenTeams(int BiggerTeam, int Imbalance, optional bool bNoSwitchHistory)
{
	local PlayerController PC;
	local int i, j, k, l, m, n, o, p, SwitchNum, GroupLen, StartIdx, TempInt;
	local array<PlayerController> SwitchCandidates;
	local array<SwitchGroup> ActiveSwitchGroups;
	local bool bSortOperator, bSwap;
	local array<Actor> DesiredObjectives;

	// Determine the number of players which need to be switched
	SwitchNum = (Imbalance & 0xFFFE) / 2; // 0xFFFE strips out odd values...3 becomes 2, 5 become 4 etc.

	// *****
	// TODO: Old, hard-coded switch ordering (remove this code) NOTE: You moved the imbalance code up here ^^ thus, you can't restore this
	bTODO = True;
	/*
	foreach WorldInfo.AllControllers(Class'PlayerController', PC)
	{
		if (PC.GetTeamNum() == BiggerTeam && ExitingPlayers.Find(PC) == -1 && bCanSwitchPlayer(PC))
		{
			//SwitchCandidates.AddItem(PC);

			// TODO: Test that this code properly prioritizes more recently joined players
			for (i=0; i<SwitchCandidates.Length; ++i)
			{
				if (PC.PlayerReplicationInfo.StartTime >= SwitchCandidates[i].PlayerReplicationInfo.StartTime)
				{
					SwitchCandidates.Insert(i, 1);
					SwitchCandidates[i] = PC;
					break;
				}
			}

			if (i == SwitchCandidates.Length)
				SwitchCandidates.AddItem(PC);
		}
	}


	// Now sift through the 'SwitchHistory' list for players on the bigger team, and move them to the bottom of the SwitchCandidates list
	// N.B. 'SwitchHistory' is automatically sorted based upon 'SwitchCount'
	foreach SwitchHistory(S)
	{
		if (S.Player != none && S.Player.GetTeamNum() == BiggerTeam)
		{
			SwitchCandidates.RemoveItem(S.Player);
			SwitchCandidates.AddItem(S.Player);
		}
	}
	*/
	// *****


	// New configurable switch ordering

	// First setup the player list
	foreach WorldInfo.AllControllers(Class'PlayerController', PC)
	{
		if (PC.GetTeamNum() == BiggerTeam && ExitingPlayers.Find(PC) == -1 && bCanSwitchPlayer(PC))
			SwitchCandidates.AddItem(PC);
	}

	// Now start ordering the players in the list

	// TODO: Make sure to add early-exit code, when the start of the topmost group is greater than the number of requires switch candidates
	bTODO = True;

	// Setup the base switch group
	ActiveSwitchGroups.Length = 1;

	ActiveSwitchGroups[0].StartIdx = 0;
	ActiveSwitchGroups[0].EndIdx = SwitchCandidates.Length-1;


	// *****
	// TODO: Remove this debug code
	bTODO = True;
	LastSwitchGroupPlayers = SwitchCandidates;

	LastSwitchGroupTree.Length = 0;
	LastSwitchGroupTree.Length = 1;
	LastSwitchGroupTree[0].Columns.Length = 1;

	LastSwitchGroupTree[0].Columns[0].StartIdx = 0;
	LastSwitchGroupTree[0].Columns[0].EndIdx = SwitchCandidates.Length-1;

	LastSwitchGroupTimestamp = WorldInfo.TimeSeconds;
	// *****


	// If flag distance is a variable that may need to be checked, then setup the values here
	for (i=0; i<SwitchOrder.Length; ++i)
	{
		if (SwitchOrder[i].Variable == SV_FlagDistance)
		{
			GetTeamObjectiveList(DesiredObjectives, BiggerTeam);
			break;
		}
	}

	// Now start iterating the active switch groups
	for (i=0; ActiveSwitchGroups.Length>0 && i<SwitchOrder.Length; ++i)
	{
		// *****
		// TODO: When the debug code is gone, move this if statement into the for loop
		bTODO = True;
		//if (ActiveSwitchGroups[0].StartIdx<j) {}
		// *****

		// Start ordering the players in the group, and create new groups as necessary
		for (j=0; j<ActiveSwitchGroups.Length; ++j)
		{
			// First sort the players
			bSortOperator = SwitchOrder[i].Placement == SP_High;


			// *****
			bTODO = True;
			// TODO: Add SwitchVariable elements based upon the EShuffleMode struct (will need to grab data from the TTFGeneric class)
			// *****

			StartIdx = ActiveSwitchGroups[j].StartIdx;
			GroupLen = (ActiveSwitchGroups[j].EndIdx-StartIdx) + 1;

			// Fugly implementation of the shell sorting algorithm (hope I don't have to debug this later)
			for (l=StartIdx+Round(GroupLen/2.0); l>StartIdx; l=StartIdx+Round((l-StartIdx)/2.0))
			{
				for (m=l; m<StartIdx+GroupLen; ++m)
				{
					PC = SwitchCandidates[m];

					for (n=m; n>=l; n-=l)
					{
						switch (SwitchOrder[i].Variable)
						{
							case SV_SwitchCount:
								o = SwitchHistory.Find('Player', SwitchCandidates[n-l]);
								p = SwitchHistory.Find('Player', PC);

								bSwap = (o!=-1 ? int(SwitchHistory[o].SwitchCount) : 0) > (p!=-1 ? int(SwitchHistory[p].SwitchCount) : 0);
								break;

							case SV_JoinTime:
								bSwap = SwitchCandidates[n-l].PlayerReplicationInfo.StartTime <
										PC.PlayerReplicationInfo.StartTime;
								break;

							case SV_Score:
								bSwap = SwitchCandidates[n-l].PlayerReplicationInfo.Score >
										PC.PlayerReplicationInfo.Score;
								break;

							case SV_Deaths:
								bSwap = SwitchCandidates[n-l].PlayerReplicationInfo.Deaths <
										PC.PlayerReplicationInfo.Deaths;
								break;

							case SV_Health:
								o = ((!SwitchCandidates[n-l].IsDead() && SwitchCandidates[n-l].Pawn != none) ?
									(SwitchCandidates[n-l].Pawn.Health +
									UTPawn(SwitchCandidates[n-l].Pawn).GetShieldStrength()) : 0);
								p = ((!PC.IsDead() && PC.Pawn != none) ? (PC.Pawn.Health + UTPawn(PC.Pawn).GetShieldStrength()) : 0);

								bSwap = o > p;
								break;

							case SV_KillingSpree:
								bSwap = UTPlayerReplicationInfo(SwitchCandidates[n-l].PlayerReplicationInfo).Spree >
										UTPlayerReplicationInfo(PC.PlayerReplicationInfo).Spree;
								break;

							case SV_HasSuperWeapon:
								bSwap = bHasSuperPickup(SwitchCandidates[n-l], True) && !bHasSuperPickup(PC, True);
								break;

							case SV_HasSuperPickup:
								bSwap = bHasSuperPickup(SwitchCandidates[n-l]) && !bHasSuperPickup(PC);
								break;

							case SV_InVehicle:
								o = int(UTVehicle(SwitchCandidates[n-l].Pawn) != none && UTVehicle_Hoverboard(SwitchCandidates[n-l].Pawn) == none);
								p = int(UTVehicle(PC.Pawn) != none && UTVehicle_Hoverboard(PC.Pawn) == none);

								bSwap = o > p;
								break;

							case SV_AliveTime:
								o = (SwitchCandidates[n-l].Pawn != none ? int(WorldInfo.TimeSeconds - SwitchCandidates[n-l].Pawn.SpawnTime) : 0);
								p = (PC.Pawn != none ? int(WorldInfo.TimeSeconds - PC.Pawn.SpawnTime) : 0);

								bSwap = o > p;
								break;

							case SV_Admin:
								bSwap = SwitchCandidates[n-l].PlayerReplicationInfo.bAdmin && !PC.PlayerReplicationInfo.bAdmin;
								break;

							case SV_FlagDistance:
								o = NearestObjectiveDist(SwitchCandidates[n-l], DesiredObjectives);
								p = NearestObjectiveDist(PC, DesiredObjectives);

								bSwap = o >= 0 && (o < p || p < 0);
								break;
						}

						if (bSwap ^^ bSortOperator)
							SwitchCandidates[n] = SwitchCandidates[n-l];
						else
							break;
					}

					SwitchCandidates[n] = PC;
				}
			}


			// Now create new groups for the sorted players

			goto 'GetValueEnd';

			// Subroutine for getting values, without another function or lots of duplicate code (input/output: k, bSortOperator, bSwap, m, n)
		GetValue:
			switch(SwitchOrder[i].Variable)
			{
				case SV_SwitchCount:
					if (bSortOperator)
					{
						bSortOperator = n != m;
					}
					else
					{
						k = SwitchHistory.Find('Player', SwitchCandidates[k]);

						if (k == -1)
							k = 0;
						else
							k = SwitchHistory[k].SwitchCount;
					}

					break;

				case SV_JoinTime:
					if (bSortOperator)
						bSortOperator = Abs(n-m) > 60;
					else
						k = SwitchCandidates[k].PlayerReplicationInfo.StartTime;

					break;

				case SV_Score:
					if (bSortOperator)
						bSortOperator = Abs(n-m) > 5;
					else
						k = SwitchCandidates[k].PlayerReplicationInfo.Score;

					break;

				case SV_Deaths:
					if (bSortOperator)
						bSortOperator = Abs(n-m) > 3;
					else
						k = SwitchCandidates[k].PlayerReplicationInfo.Deaths;

					break;

				case SV_Health:
					if (bSortOperator)
					{
						bSortOperator = Abs(n-m) > 25;
					}
					else
					{
						k = ((!SwitchCandidates[k].IsDead() && SwitchCandidates[k].Pawn != none) ? (SwitchCandidates[k].Pawn.Health +
							UTPawn(SwitchCandidates[k].Pawn).GetShieldStrength()) : 0);
					}

					break;

				case SV_KillingSpree:
					if (bSortOperator)
						bSortOperator = Abs(n-m) > 4;
					else
						k = UTPlayerReplicationInfo(SwitchCandidates[k].PlayerReplicationInfo).Spree;

					break;

				case SV_HasSuperWeapon:
					if (bSortOperator)
						bSortOperator = n != m;
					else
						k = int(bHasSuperPickup(SwitchCandidates[k], True));

					break;

				case SV_HasSuperPickup:
					if (bSortOperator)
						bSortOperator = n != m;
					else
						k = int(bHasSuperPickup(SwitchCandidates[k]));

					break;

				case SV_InVehicle:
					if (bSortOperator)
						bSortOperator = n != m;
					else
						k = int(UTVehicle(SwitchCandidates[k].Pawn) != none && UTVehicle_Hoverboard(SwitchCandidates[k].Pawn) == none);

					break;

				case SV_AliveTime:
					if (bSortOperator)
						bSortOperator = Abs(n-m) > 45;
					else
						k = (SwitchCandidates[k].Pawn != none ? int(WorldInfo.TimeSeconds - SwitchCandidates[k].Pawn.SpawnTime) : 0);

					break;

				case SV_Admin:
					if (bSortOperator)
						bSortOperator = n != m;
					else
						k = int(SwitchCandidates[k].PlayerReplicationInfo.bAdmin);

					break;

				case SV_FlagDistance:
					if (bSortOperator)
						bSortOperator = Abs(n-m) > 2000 || (n == -1 ^^ m == -1);
					else
						k = NearestObjectiveDist(SwitchCandidates[k], DesiredObjectives);

					break;
			}

			if (l == -1)
				goto 'PreLoop';
			else if (bSwap)
				goto 'LoopOperator';
			else
				goto 'LoopStart';
		GetValueEnd:


			// ***
			// m = GetValue(StartIdx);
			l = -1;
			bSwap = False;
			bSortOperator = False;
			k = StartIdx;
			goto 'GetValue';
		PreLoop:
			m = k;
			// ***

			for (l=0; l<GroupLen; ++l)
			{
				// ***
				// n = GetValue(StartIdx+l);
				bSwap = False;
				bSortOperator = False;
				k = StartIdx+l;
				goto 'GetValue';
			LoopStart:
				n = k;
				// ***

				// ***
				bSwap = True;
				bSortOperator = True;
				goto 'GetValue';
			LoopOperator:
				// ***

				// New Group
				if (bSortOperator)
				{
					// Cap the old group
					ActiveSwitchGroups[j].EndIdx = StartIdx+l-1;

					// Reuse the old group if necessary
					if (ActiveSwitchGroups[j].EndIdx - ActiveSwitchGroups[j].StartIdx < 1)
						o = j;
					else
						o = -1;

					// Setup the new group, if its big enough
					if (GroupLen-l >= 2)
					{
						if (o == -1)
						{
							o = ++j;
							ActiveSwitchGroups.Insert(j, 1);
						}

						ActiveSwitchGroups[o].StartIdx = StartIdx+l;
						ActiveSwitchGroups[o].EndIdx = StartIdx+GroupLen-1;
					}
					// Remove the group being replaced, if none are getting added (end of loop)
					else if (o != -1)
					{
						ActiveSwitchGroups.Remove(j--, 1);
						break;
					}

					// *****
					bTODO = True;
					// TODO: Remove this debug code

					// Expand the debug tree
					if (LastSwitchGroupTree.Length < i+2)
						LastSwitchGroupTree.Length = i+2;

					// Identify the parent tree element
					for (k=0; k<LastSwitchGroupTree[i].Columns.Length; ++k)
						if (LastSwitchGroupTree[i].Columns[k].StartIdx == StartIdx)
							break;

					// Add the new element, and add its index to the parent tree subgroup list
					o = LastSwitchGroupTree[i+1].Columns.Add(1);
					LastSwitchGroupTree[i].Columns[k].SubGroups.AddItem(o);

					LastSwitchGrouptree[i+1].Columns[o].StartIdx = StartIdx+l;
					LastSwitchGroupTree[i+1].Columns[o].EndIdx = StartIdx+GroupLen-1;
					LastSwitchGroupTree[i+1].Columns[o].SwitchInfo = SwitchOrder[i];
					LastSwitchGroupTree[i+1].Columns[o].bPostEarlyExit = ActiveSwitchGroups[0].StartIdx>=j;
					// *****

					m = n;
				}
			}
		}
	}


	SwitchCandidates.Length = SwitchNum;

	NotifySwitchPlayers(SwitchCandidates);

	for (i=0; i<SwitchCandidates.Length; ++i)
		SwitchPlayer(SwitchCandidates[i], bNoSwitchHistory);


	// If the teams are still imbalanced by raw number, then rebalance the bots
	// TODO
}

// Returns true if it's ok to switch this player midgame
function bool bCanSwitchPlayer(PlayerController PC)
{
	local UTPlayerReplicationInfo PRI;

	PRI = UTPlayerReplicationInfo(PC.PlayerReplicationInfo);
	return (PRI != none && !PRI.bOnlySpectator && !PRI.bHasFlag && !PRI.IsHero() && PC.HasClientLoadedCurrentWorld());
}

function SwitchPlayer(PlayerController PC, optional bool bNoSwitchHistory)
{
	local int i, NewPos;
	local SwitchList NewEntry, SH;

	// If there is to be no history change, then just redirect to the changeteam function directly
	if (bNoSwitchHistory)
	{
		TTFChangeTeam(PC, int(!bool(PC.GetTeamNum())));
		return;
	}


	// Check if the current controller is in the switch history list (cleaning the list as you go).
	/*
	CurEntry = -1;
	NewPos = SwitchHistory.Length;

	for (i=0; i<SwitchHistory.Length; ++i)
	{
		// Remove null entries (making sure to reduce NewPos as well)
		if (SwitchHistory[i].Player == none)
		{
			SwitchHistory.Remove(i, 1);
			--NewPos;
			--i;

			continue;
		}


		if (CurEntry == -1)
		{
			if (SwitchHistory[i].Player == C)
				CurEntry = i;
		}
		// If the current controller has been found in the list, try to find a place to reposition him whilst the list continues to iterate
		else if (NewPos == SwitchHistory.Length)
		{
			// ***
			bRemoveDebugCode = True;
			//LogInternal("TTeamFix::SwitchPlayer: Searching for suitable repositioning spot in SwitchHistory", 'TTFDebug');

			if (bDebugLogging)
				DebugLog.Logf("TTFDebug: TTeamFix::SwitchPlayer: Searching for suitable repositioning spot in SwitchHistory");
			// ***

			// Order by SwitchCount first, and then by StartTime
			if (SwitchHistory[CurEntry].SwitchCount + 1 > SwitchHistory[i].SwitchCount
				|| C.PlayerReplicationInfo.StartTime >= SwitchHistory[i].Player.PlayerReplicationInfo.StartTime)
			{
				// ***
				bRemoveDebugCode = True;

				/*
				LogInternal("TTeamFix::SwitchPlayer: Determined new repositioning spot; compared variables were:", 'TTFDebug');
				LogInternal("---", 'TTFDebug');
				LogInternal("SwitchCount:"@(SwitchHistory[CurEntry].SwitchCount + 1)@">"@SwitchHistory[i].SwitchCount, 'TTFDebug');
				LogInternal("StartTime:"@C.PlayerReplicationInfo.StartTime@">="@SwitchHistory[i].Player.PlayerReplicationInfo.StartTime, 'TTFDebug');
				LogInternal("---", 'TTFDebug');
				*/

				if (bDebugLogging)
				{
					DebugLog.Logf("TTFDebug: TTeamFix::SwitchPlayer: Determined new repositioning spot; compared variables were:");
					DebugLog.Logf("TTFDebug: ---");
					DebugLog.Logf("TTFDebug: SwitchCount:"@(SwitchHistory[CurEntry].SwitchCount + 1)@">"@SwitchHistory[i].SwitchCount);
					DebugLog.Logf("TTFDebug: StartTime:"@C.PlayerReplicationInfo.StartTime@">="@SwitchHistory[i].Player.PlayerReplicationInfo.StartTime);
					DebugLog.Logf("TTFDebug: ---");
				}
				// ***

				NewPos = i;
			}
		}
	}

	NewEntry.Player = C;
	NewEntry.SwitchCount = (CurEntry == -1) ? 1 : SwitchHistory[CurEntry].SwitchCount;
	*/


	// There is too much potential for bugs in the above code; this is less efficient but keeps things simple (perhaps recode later)


	// First clean the 'SwitchHistory' list
	i = SwitchHistory.Find('Player', None);

	while (i != -1)
	{
		SwitchHistory.Remove(i, 1);
		i = SwitchHistory.Find('Player', None);
	}


	// Now check if the current controller is already in the list (removing if found) and setup the new entry data
	i = SwitchHistory.Find('Player', PC);

	NewEntry.Player = PC;

	if (i == -1)
	{
		NewEntry.SwitchCount = 1;
	}
	else
	{
		NewEntry.SwitchCount = SwitchHistory[i].SwitchCount + 1;
		SwitchHistory.Remove(i, 1);
	}


	// Find a suitable location for the controller within the list
	NewPos = -1;

	foreach SwitchHistory(SH, i)
	{
		// Sorted first by SwitchCount and then StartTime
		if (NewEntry.SwitchCount < SH.SwitchCount ||
			(NewEntry.SwitchCount == SH.SwitchCount && PC.PlayerReplicationInfo.StartTime >= SH.Player.PlayerReplicationInfo.StartTime))
		{
			NewPos = i;
			break;
		}
	}

	// If a suitable location was not found, add to the end of the list, otherwise insert directly
	if (NewPos == -1)
	{
		NewPos = SwitchHistory.Length;
		SwitchHistory.Length = NewPos + 1;

		SwitchHistory[NewPos] = NewEntry;
	}
	else
	{
		SwitchHistory.InsertItem(NewPos, NewEntry);
	}



	// Switch the player
	TTFChangeTeam(PC, int(!bool(PC.GetTeamNum())));
}


// ===== Cannibalized versions of team-changing functions defined in gameinfo and its subclasses

function TTFChangeTeam(Controller Other, int num)
{
	local bool bOldKillEvent;
	local PlayerController PC;
	local UTOnslaughtGame OG;
	local UTOnslaughtPRI PRI;
	local UTPlayerReplicationInfo DefPRI;
	local int i, OldScore, OldDeaths, OldLives, OldSpree;
	local bool bOldOutOfLives;

	// This deprecated debug msg is here to remind me to check the latest patch code to see if any of this ripped code has changed
	bCheckAgainstPatch_LastVer_Patch4 = True;


	// Very important: You need to temporarily disable kill events, otherwise PlayerKilled could attempt to balance teams while this is calling
	bOldKillEvent = EventRules.bKillEvent;
	EventRules.bKillEvent = False;



	TTFSetTeam(Other, UTTeamGame(WorldInfo.Game).Teams[num]);


	OG = UTOnslaughtGame(WorldInfo.Game);

	if (OG != none)
	{
		foreach LocalPlayerControllers(Class'PlayerController', PC)
		{
			if (Other == PC)
				for (i=0; i<OG.PowerNodes.Length; ++i)
					OG.PowerNodes[i].UpdateEffects(False);

			break;
		}


		PRI = UTOnslaughtPRI(Other.PlayerReplicationInfo);

		if (PRI != none)
		{
			PRI.StartObjective = None;
			PRI.TemporaryStartObjective = None;
		}
	}


	if (Other.Pawn != none)
	{
		DefPRI = UTPlayerReplicationInfo(Other.PlayerReplicationInfo);

		OldScore =		DefPRI.Score;
		OldDeaths =		DefPRI.Deaths;
		OldLives =		DefPRI.NumLives;	// dunno if this might cause bugs...but it's no worse than leaving it out altogether
		bOldOutOfLives =	DefPRI.bOutOfLives;	// ^^
		OldSpree =		DefPRI.Spree;

		Other.Pawn.PlayerChangedTeam();

		DefPRI.Score =		OldScore;
		DefPRI.Deaths =		OldDeaths;
		DefPRI.NumLives =	OldLives;
		DefPRI.bOutOfLives =	bOldOutOfLives;
		DefPRI.Spree =		OldSpree;	
	}


	// Reset kill events now that the player has been switched (N.B. Keep this at the bottom of the function)
	EventRules.bKillEvent = bOldKillEvent;
}

function TTFSetTeam(Controller Other, UTTeamInfo NewTeam)
{
	local UTPlayerReplicationInfo PRI;
	local actor A;

	// This deprecated debug msg is here to remind me to check the latest patch code to see if any of this ripped code has changed
	bCheckAgainstPatch_LastVer_Patch4 = True;


	if (Other.PlayerReplicationInfo == none)
		return;

	if (Other.PlayerReplicationInfo.Team != none || !WorldInfo.Game.ShouldSpawnAtStartSpot(Other))
		Other.StartSpot = None;


	if (WorldInfo.NetMode != NM_DedicatedServer)
	{
		PRI = UTPlayerReplicationInfo(Other.PlayerReplicationInfo);

		if (PRI != none && !PRI.IsLocalPlayerPRI())
			PRI.SetCharacterMesh(None);
	}

	if (Other.PlayerReplicationInfo.Team != none)
	{
		Other.PlayerReplicationInfo.Team.RemoveFromTeam(Other);
		Other.PlayerReplicationInfo.Team = none;
	}

	if (NewTeam != none)
		NewTeam.AddToTeam(Other);


	if (PlayerController(Other) != none && LocalPlayer(PlayerController(Other).Player) != none)
		foreach AllActors(Class'Actor', A)
			A.NotifyLocalPlayerTeamReceived();

	if (WorldInfo.NetMode != NM_DedicatedServer && PRI != none && UTGameReplicationInfo(WorldInfo.GRI) != none)
		UTGameReplicationInfo(WorldInfo.GRI).ProcessCharacterData(PRI, True);
}


defaultproperties
{
	EventMutClass=Class'TTeamFixMut'
	EventRulesClass=Class'TTeamFixRules'

	RecentCutoffTime=120
}