//============================================================
// TTeamFixQueryHandler_Generic.uc	- Manages webadmin configuration for TitanTeamFix
//============================================================
//	TitanTeamFix
//		- A modular team balancing tool initially coded for the Titan servers:
//			http://ut2004.titaninternet.co.uk/
//
//	Copyright (C) 2007-2009 John "Shambler" Barrett (JBarrett847@Gmail.com or Shambler@OldUnreal.com)
//
//	This program is free software; you can redistribute and/or modify
//	it under the terms of the Open Unreal Mod License version 1.1.
//
//============================================================
//
// Handles setup of and query requests from the TTF configuration page within the WebAdmin
//
//============================================================
Class TTeamFixQueryHandler_Generic extends TTeamFixQueryHandler
	implements(IQueryHandler);

var WebAdmin				WebAdmin;		// Cached instance of the webadmin object
var array<TTeamFixConfigProfile>	ConfigProfiles;		// Currently, only config profiles of the active team balancer are added

var TTeamFixMutSettings			TTFSettingsObj;		// An instance of the class which handles TTF's main settings

var TTeamFixSettingsRenderer		TTFSettingsRenderer;	// A settings renderer specifically for TTF config profiles

var deprecated bool			bTODO;


function Init(WebAdmin WebApp)
{
	local array<string> ConfigSections;
	local int i, ConfigClassLen;
	local string CurSectionObjName;

	// ***
	bTODO = True;
	// Add property grouping to the generic balancer settings, if possible
	// UPDATE: This is low priority, maybe another time
	// ***

	WebAdmin = WebApp;

	if (TeamBalancer != none && GetPerObjectConfigSections(TeamBalancer.ConfigProfileClass, ConfigSections))
	{
		ConfigProfiles.Length = ConfigSections.Length;
		ConfigClassLen = Len(string(TeamBalancer.ConfigProfileClass.Name)) + 1;

		for (i=0; i<ConfigSections.Length; ++i)
		{
			CurSectionObjName = Left(ConfigSections[i], Len(ConfigSections[i])-ConfigClassLen);
			ConfigProfiles[i] = Class'TTeamFixConfigLoader'.static.FindConfigProfile(CurSectionObjName, TeamBalancer.ConfigProfileClass);

			ConfigProfiles[i].InitWebAdminProperties(TeamBalancer);
		}
	}


	TTFSettingsObj = new Class'TTeamFixMutSettings';
	TTFSettingsObj.SetSpecialValue('WebAdmin_Init', "");

	TTFSettingsRenderer = new Class'TTeamFixSettingsRenderer';
	TTFSettingsRenderer.Init(WebAdmin.Path);
}

// Clean up any actor and other references etc. here
function Cleanup()
{
	WebAdmin = none;
	ConfigProfiles.Length = 0;
	TTFSettingsObj = none;
	TTFSettingsRenderer = none;
}


function bool HandleQuery(WebAdminQuery Q)
{
	switch (Q.Request.URI)
	{
		case "/TitanTeamFix":
			HandleTTFConfigQuery(Q);
			return True;
	}

	return False;
}

function bool UnhandledQuery(WebAdminQuery Q)
{
	return False;
}

function RegisterMenuItems(WebAdminMenu Menu)
{
	Menu.AddMenu("/TitanTeamFix", "Titan Team Fix", Self, "Titan Team Fix settings", 1);
}


function HandleTTFConfigQuery(WebAdminQuery Q)
{
	// TODO: Cut down on the number of these variables
	local string CurEditProfile, CurAction, SafeStr, NewProfile, NewProfileGame, Warning, Error;
	local array<string> SubStrEntities;
	local int i;
	local TTeamFixConfigProfile CurEditProfileObj;
	local bool bOldWebAdminOption;
	local UTUIDataProvider_GameModeInfo GameDP;

	CurEditProfile = Q.Request.GetVariable("ProfileName");

	if (CurEditProfile == "" && TeamBalancer.ConfigObject != none)
		CurEditProfile = string(TeamBalancer.ConfigObject.Name);

	if (CurEditProfile == "")
		CurEditProfile = "default";

	if (CurEditProfile != "")
		for (i=0; i<ConfigProfiles.Length; ++i)
			if (string(ConfigProfiles[i].Name) ~= CurEditProfile)
				CurEditProfileObj = ConfigProfiles[i];


	CurAction = Q.Request.GetVariable("action");


	if (CurAction ~= "TTFMainSave")
	{
		bOldWebAdminOption = Class'TTeamFixConfigLoader'.default.bAddWebAdminConfigMenu;

		ApplySettings(TTFSettingsObj, Q.Request);
		TTFSettingsObj.SetSpecialValue('WebAdmin_Save', "");

		WebAdmin.AddMessage(Q, "Titan Team Fix settings saved");

		if (bOldWebAdminOption && !Class'TTeamFixConfigLoader'.default.bAddWebAdminConfigMenu)
			WebAdmin.AddMessage(Q, "NOTE: You can re-enable the Titan Team Fix WebAdmin menu through the main mutator settings");
	}
	else if (CurAction ~= "TTFProfileCreate")
	{
		NewProfile = Q.Request.GetVariable("NewProfileName");
		NewProfileGame = Q.Request.GetVariable("NewProfileGameClass");

		if (NewProfile != "")
		{
			if (NewProfileGame != "")
				NewProfile $= "_"$NewProfileGame;

			for (i=0; i<ConfigProfiles.Length; ++i)
				if (string(ConfigProfiles[i].Name) ~= NewProfile)
					break;


			if (!IsAlphaNumeric(NewProfile, "_-"))
			{
				WebAdmin.AddMessage(Q, "Only characters A-Z, a-z, 0-9, '_' and '-' are valid", MT_Error);
			}
			else if (i == ConfigProfiles.Length)
			{
				CurEditProfileObj = Class'TTeamFixConfigLoader'.static.FindConfigProfile(NewProfile, TeamBalancer.ConfigProfileClass);
				CurEditProfileObj.SaveConfig();
				CurEditProfileObj.InitWebAdminProperties(TeamBalancer);

				ConfigProfiles.AddItem(CurEditProfileObj);

				WebAdmin.AddMessage(Q, "Created configuration profile '"$CurEditProfileObj.Name$"'");
			}
			else
			{
				WebAdmin.AddMessage(Q, "There's already a profile with the name"@NewProfile, MT_Error);
			}
		}
		else
		{
			WebAdmin.AddMessage(Q, "Please enter a profile name", MT_Error);
		}
	}
	else if (CurAction ~= "TTFProfileSave")
	{
		if (CurEditProfileObj != none)
		{
			ApplySettings(CurEditProfileObj, Q.Request);
			CurEditProfileObj.SaveWebAdminProperties(TeamBalancer, Warning, Error);

			WebAdmin.AddMessage(Q, "Titan Team Fix configuration profile '"$CurEditProfileObj.Name$"' saved");

			if (Warning != "")
				WebAdmin.AddMessage(Q, Warning, MT_Warning);

			if (Error != "")
				WebAdmin.AddMessage(Q, Error, MT_Error);
		}
		else
		{
			WebAdmin.AddMessage(Q, "Couldn't find configuration profile which was being edited, profile not saved", MT_Error);
		}
	}
	else if (CurAction ~= "TTFProfileDelete")
	{
		if (CurEditProfileObj != none)
		{
			if (!(CurEditProfileObj.Name == 'default'))
			{
				ConfigProfiles.RemoveItem(CurEditProfileObj);
				CurEditProfileObj.ClearConfig();

				if (TeamBalancer.ConfigObject == CurEditProfileObj)
					TeamBalancer.ConfigObject = none;

				WebAdmin.AddMessage(Q, "Titan Team Fix configuration profile '"$CurEditProfileObj.Name$"' has been deleted");

				for (i=0; i<ConfigProfiles.Length; ++i)
					if (ConfigProfiles[i].Name == 'default')
						CurEditProfileObj = ConfigProfiles[i];
			}
			else
			{
				WebAdmin.AddMessage(Q, "You can't delete the default configuration profile", MT_Error);
			}
		}
		else
		{
			WebAdmin.AddMessage(Q, "Couldn't find configuration profile which was being edited, profile not deleted", MT_Error);
		}
	}


	// Build and apply the main settings substitute string
	TTFSettingsRenderer.Render(TTFSettingsObj, Q.Response, "TTFMainSettings");


	// Build and apply the config profile list subtitute string
	SubStrEntities[0] = "";

	for (i=0; i<ConfigProfiles.Length; ++i)
	{
		SafeStr = Class'WebAdminUtils'.static.HTMLEscape(ConfigProfiles[i].Name);
		Q.Response.Subst("profile.id", SafeStr);

		if (ConfigProfiles[i] == CurEditProfileObj)
			Q.Response.Subst("profile.selected", "selected=\"selected\"");
		else
			Q.Response.Subst("profile.selected", "");

		if (TeamBalancer.ConfigObject == ConfigProfiles[i])
			Q.Response.Subst("profile.displayname", SafeStr@"(currently in use)");
		else
			Q.Response.Subst("profile.displayname", SafeStr);

		SubStrEntities[0] $= WebAdmin.Include(Q, "TitanTeamFix_Profile_Select.inc");
	}

	Q.Response.Subst("TTFProfiles", SubStrEntities[0]);


	if (Class'TTeamFixConfigLoader'.default.bSuffixGameToProfile)
	{
		SubStrEntities[0] = "";

		// First add a blank entry
		Q.Response.Subst("gametype.gamemode", "");
		Q.Response.Subst("gametype.friendlyname", "");
		Q.Response.Subst("gametype.selected", "selected=\"selected\"");
		SubStrEntities[0] $= WebAdmin.Include(Q, "current_change_gametype.inc");


		WebAdmin.DataStoreCache.LoadGameTypes();

		foreach WebAdmin.DataStoreCache.GameTypes(GameDP)
		{
			if (GameDP.bIsCampaign)
				continue;


			NewProfileGame = GameDP.GameMode;
			i = InStr(NewProfileGame, ".");

			if (i != INDEX_None)
				NewProfileGame = Mid(NewProfileGame, i+1);

			Q.Response.Subst("gametype.gamemode", Class'WebAdminUtils'.static.HTMLEscape(NewProfileGame));
			Q.Response.Subst("gametype.friendlyname", Class'WebAdminUtils'.static.HTMLEscape(Class'WebAdminUtils'.static.GetLocalized(
									GameDP.FriendlyName)));
			Q.Response.Subst("gametype.selected", "");

			SubStrEntities[0] $= WebAdmin.Include(Q, "current_change_gametype.inc");
		}

		Q.Response.Subst("TTFGameClasses", SubStrEntities[0]);

		SubStrEntities[1] = WebAdmin.Include(Q, "TitanTeamFix_NewProfile_Game.inc");

		Q.Response.Subst("TTFProfileGameSelection", SubStrEntities[1]);
	}
	else
	{
		Q.Response.Subst("TTFProfileGameSelection", "");
	}


	if (CurEditProfileObj != none)
	{
		SubStrEntities[0] = "";

		Q.Response.Subst("ProfileName", CurEditProfileObj.Name);
		Q.Response.Subst("TTFProfile.ProfileName", CurEditProfileObj.Name);
		TTFSettingsRenderer.Render(CurEditProfileObj, Q.Response, "TTFProfileSettings");

		SubStrEntities[0] $= WebAdmin.Include(Q, "TitanTeamFix_Profile_Editor.inc");

		Q.Response.Subst("ProfileEditor", SubStrEntities[0]);
	}
	else
	{
		Q.Response.Subst("ProfileEditor", "");
	}


	WebAdmin.SendPage(Q, "TitanTeamFix_Config.html");
}


// ***** Functions ripped directly from the WebAdmin

// It would be better if this were already static/final in the webadmin code
static final function ApplySettings(Settings settings, WebRequest request, optional string prefix = "settings_")
{
	local int i, idx;
	local name sname;
	local string val;

	for (i = 0; i < settings.LocalizedSettingsMappings.Length; i++)
	{
		idx = settings.LocalizedSettingsMappings[i].Id;
		sname = settings.GetStringSettingName(idx);
		if (request.GetVariableCount(prefix$sname) > 0)
		{
			val = request.GetVariable(prefix$sname);
			settings.SetStringSettingValue(idx, int(val), false);
		}
	}
	for (i = 0; i < settings.PropertyMappings.Length; i++)
	{
		idx = settings.PropertyMappings[i].Id;
		sname = settings.GetPropertyName(idx);
		if (request.GetVariableCount(prefix$sname) > 0)
		{
			val = request.GetVariable(prefix$sname);
			settings.SetPropertyFromStringByName(sname, val);
		}
	}
}


