/* 
    BattleRPG Copyright (C) 2007 Nico de Vries.

    This file is part of BattleRPG.

    BattleRPG is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see http://www.gnu.org/licenses/.
*/

class BattleStaticFunctions extends Object;
// Most of the functions here simulate multiple inheritance. An alternative would have been to attach
// items to other objects like BattleGenInvItem does it, but this aproach is more efficient and probably 
// easier to maintain. Especially for distributed data. It only works for straihtforward methods (functions) though.

var const color Black, White, Gold;
 
// Simulate multiple inheritance by adding this data to UTPlayerReplicationInfo, UTDuelPRI and UTOnslaughtPRI
struct BPRIDATA {
// Client + server (replicated), this data will be replicated for all clients to all clients, so everyone sees everything
  var int Level; 
  var int XP;
  var int NeededXP;
  var int Mana;

// Server only
  var int PrevScore;

// Make sure we can reach this from everywhere
  var BattlePersistentPlayerDatabase BPD;
};

static simulated function BPRIDATA ReadBPRI (PlayerReplicationInfo PRI)
{
  local BPRIDATA Result;
  if (BattlePRIONS (PRI) != None) {
    Result.Level = BattlePRIONS (PRI).Level;
    Result.XP = BattlePRIONS (PRI).XP;
    Result.NeededXP = BattlePRIONS (PRI).NeededXP;
    Result.Mana = BattlePRIONS (PRI).Mana;
    Result.PrevScore = BattlePRIONS (PRI).PrevScore;
  } else if (BattlePRI (PRI) != None) {
    Result.Level = BattlePRI (PRI).Level;
    Result.XP = BattlePRI (PRI).XP;
    Result.NeededXP = BattlePRI (PRI).NeededXP;
    Result.Mana = BattlePRI (PRI).Mana;
    Result.PrevScore = BattlePRI (PRI).PrevScore;
  } else if (BattlePRIDuel (PRI) != None) {
    Result.Level = BattlePRIDuel (PRI).Level;
    Result.XP = BattlePRIDuel (PRI).XP;
    Result.NeededXP = BattlePRIDuel (PRI).NeededXP;
    Result.Mana = BattlePRIDuel (PRI).Mana;
    Result.PrevScore = BattlePRIDuel (PRI).PrevScore;
  }
  return Result;
}

static simulated function WriteBPRI (PlayerReplicationInfo PRI, BPRIDATA BD)
{
  local bool Dirty;
  // Minimize network overhead by only processing changes and using small units
  Dirty = false;
  if (BattlePRIONS (PRI) != None) {
    if (BattlePRIONS (PRI).Level != BD.Level) BattlePRIONS (PRI).Level = BD.Level;
    if (BattlePRIONS (PRI).XP != BD.XP) {
      BattlePRIONS (PRI).XP = BD.XP;
      Dirty = true;
    }
    if (BattlePRIONS (PRI).NeededXP != BD.NeededXP) BattlePRIONS (PRI).NeededXP = BD.NeededXP;
    if (BattlePRIONS (PRI).Mana != BD.Mana) BattlePRIONS (PRI).Mana = BD.Mana;
    BattlePRIONS (PRI).PrevScore = BD.PrevScore; // Don't waste CPU cycles
    if (BD.BPD != None) BattlePRIONS (PRI).BPD = BD.BPD; // Make sure we can always access this
  } else if (BattlePRI (PRI) != None) {
    if (BattlePRI (PRI).Level != BD.Level) BattlePRI (PRI).Level = BD.Level;
    if (BattlePRI (PRI).XP != BD.XP) {
      BattlePRI (PRI).XP = BD.XP;
      Dirty = true;
    }
    if (BattlePRI (PRI).NeededXP != BD.NeededXP) BattlePRI (PRI).NeededXP = BD.NeededXP;
    if (BattlePRI (PRI).Mana != BD.Mana) BattlePRI (PRI).Mana = BD.Mana;
    BattlePRI (PRI).PrevScore = BD.PrevScore;
    if (BD.BPD != None) BattlePRI (PRI).BPD = BD.BPD; // Make sure we can always access this
  } else if (BattlePRIDuel (PRI) != None) {
    if (BattlePRIDuel (PRI).Level != BD.Level) BattlePRIDuel (PRI).Level = BD.Level;
    if (BattlePRIDuel (PRI).XP != BD.XP) {
      BattlePRIDuel (PRI).XP = BD.XP;
      Dirty = true;
    }
    if (BattlePRIDuel (PRI).NeededXP != BD.NeededXP) BattlePRIDuel (PRI).NeededXP = BD.NeededXP;
    if (BattlePRIDuel (PRI).Mana != BD.Mana) BattlePRIDuel (PRI).Mana = BD.Mana;
    BattlePRIDuel (PRI).PrevScore = BD.PrevScore;
    if (BD.BPD != None) BattlePRIDuel(PRI).BPD = BD.BPD; // Make sure we can always access this
  }
  if (Dirty) {
    if (BattlePRIDuel(PRI) != None && BattlePRIDuel(PRI).BPD != None) {
      BattlePRIDuel(PRI).BPD.SavePlayerData (PRI); // Save data to persistent database as well
    }
    if (BattlePRI(PRI) != None && BattlePRI(PRI).BPD != None) {
      BattlePRI(PRI).BPD.SavePlayerData (PRI); // Save data to persistent database as well
    }
    if (BattlePRIONS(PRI) != None && BattlePRIONS(PRI).BPD != None) {
      BattlePRIONS(PRI).BPD.SavePlayerData (PRI); // Save data to persistent database as well
    }
  }
}

// With multiple inheritance this might have been a method of UTPlayerReplicationInfo, UTDuelPRI and UTOnslaughtPRI
static simulated function PRITick (UTPlayerReplicationInfo PRI)
{
  local int Score;
  local BPRIDATA BD;

  Score = int(PRI.Score);
  BD = class'BattleStaticFunctions'.static.ReadBPRI (PRI);
  if (BD.Level == 0 || Score != BD.PrevScore) {
    if (BD.Level == 0) { // Initialize RPG data
      BD.Level = 1;
    }
    if (BD.NeededXP == 0) { // Initialize RPG data
      BD.NeededXP = 2;
    }

    // Prototype RPG scoring system, this needs to become configurable and overridable
    // The system is easy, for every point you get you also get 1 XP
    // The first 10 levels are very easy to get players to learn how RPG works
    // Reaching level 80 requires some serious playing but is doable
    // Level 100 is tougher to reach
    // Over level 100 it becomes extremely hard
    if (Score > BD.PrevScore) {
      BD.XP += Score - BD.PrevScore;
      while (BD.XP >= BD.NeededXP) {
        BD.XP -= BD.NeededXP;
        BD.Level++;
        if (BD.Level <= 10) {
          BD.NeededXP = 2 * BD.Level; // 2, 4, .. 20
        } else if (BD.Level <= 80) {
          BD.NeededXP = BD.Level * 10 - 90; // 30, 40, .. 710
        } else if (BD.Level < 100) {
          BD.NeededXP = BD.Level * 100 - 7100; // 1000, 1100, .. 
        } else if (BD.Level < 110) {
          BD.NeededXP = BD.Level * 1000; 
        } else if (BD.Level < 120) {
          BD.NeededXP = BD.Level * 10000;
        } else if (BD.Level < 130) {
          BD.NeededXP = BD.Level * 100000;
        } else {
          BD.NeededXP = BD.Level * 1000000;
        }
      } 
    }

    // Prototype Mana scoring system, this needs to become configurable and overridable
    if (Score > BD.PrevScore) {
      BD.Mana += 2 + (Score-BD.PrevScore); // Mostly less mana for achievements, more for killing
      if (BD.Mana > 100) BD.Mana = 100;
    }

    BD.PrevScore = Score;
    class'BattleStaticFunctions'.static.WriteBPRI (PRI, BD);
  }
}

// With multiple inheritance this might have been a method of UTCTFScoreboardPanel, UTDuelQueueScoreboardPanel,
// UTONSScoreboardPanel, UTScoreboardPanel and UTTDMScoreboardPanel
static simulated function string GetExtraRightMisc (UTPlayerReplicationInfo PRI, int Deaths, int Score, optional bool bIncludeDeaths=true)
{
  local int TotalSeconds, PPH;
  local BPRIDATA BD;

  BD = class'BattleStaticFunctions'.static.ReadBPRI (PRI);
  TotalSeconds = PRI.WorldInfo.GRI.ElapsedTime - PRI.StartTime;
  PPH = int(float(Score) / (float(TotalSeconds+1)/float(3600)));
  if (bIncludeDeaths) {
    return "Deaths "$Deaths$"   "$"PPH "$PPH$"   "$"Level "$BD.Level$" ("$BD.XP$"/"$BD.NeededXP$")";
  } else {
    return "PPH "$PPH$"   "$"RPG Level "$BD.Level$"("$BD.XP$"/"$BD.NeededXP$")";
  }
}

// With multiple inheritance this might have been a method of UTHUD, UTCTFHUD, UTDuelHUD, UTOnslaughtHUD and UTTeamHUD
static simulated function DrawTile (UTHUD H, String Style, float X, float Y, float Wi, float He)
{
  // Simulate a virtual 1024*768 screen  
  X = X * H.ResolutionScaleX;
  Y = Y * H.ResolutionScale;
  H.Canvas.SetPos(X, Y);
  H.Canvas.DrawColorizedTile(H.AltHudTexture, Wi * H.ResolutionScale, He * H.ResolutionScale, 489, 395,183, 44, H.TeamHudColor);
}

// With multiple inheritance this might have been a method of UTHUD, UTCTFHUD, UTDuelHUD, UTOnslaughtHUD and UTTeamHUD
static simulated function DrawText (UTHUD H, float X, float Y, string Text, optional string Font="large", optional string Style="goldshadow")
{
  if (H == None || H.Canvas == None) return;

  // Simulate a virtual 1024*768 screen  
  X = X * H.ResolutionScaleX;
  Y = Y * H.ResolutionScale;

  if (Font == "small") {
    H.Canvas.Font = class'Engine'.static.GetTinyFont();
  } else if (Font == "large") {
    H.Canvas.Font = class'Engine'.static.GetLargeFont();
  } else if (Font == "s0") {
    H.Canvas.Font = H.GetFontSizeIndex(0);
  } else if (Font == "s1") {
    H.Canvas.Font = H.GetFontSizeIndex(1);
  } else if (Font == "s2") {
    H.Canvas.Font = H.GetFontSizeIndex(2);
  }

  if (Style=="white") {
	H.Canvas.DrawColor = default.White;
    H.Canvas.SetPos (X, Y);
	H.Canvas.DrawTextClipped (Text, false);
  }
  if (Style=="shadow") {
    H.Canvas.DrawColor = default.Black;
    if (Font == "small" || Font == "s0" || Font == "s1") H.Canvas.SetPos (X+1, Y+1); else H.Canvas.SetPos (X+2, Y+2);
    H.Canvas.DrawTextClipped (Text, false);
	H.Canvas.DrawColor = default.White;
    H.Canvas.SetPos (X, Y);
	H.Canvas.DrawTextClipped (Text, false);
  }  
  if (Style=="goldshadow") {
    H.Canvas.DrawColor = default.Black;
    if (Font == "small" || Font == "s0" || Font == "s1") H.Canvas.SetPos (X+1, Y+1); else H.Canvas.SetPos (X+2, Y+2);
    H.Canvas.DrawTextClipped (Text, false);
	H.Canvas.DrawColor = default.Gold;
    H.Canvas.SetPos (X, Y);
	H.Canvas.DrawTextClipped (Text, false);
  }  
  if (Style=="aura") {
    H.Canvas.DrawColor = default.Black;
    if (Font == "small" || Font == "s0" || Font == "s1") H.Canvas.SetPos (X+1, Y+1); else H.Canvas.SetPos (X+2, Y+2);
    H.Canvas.DrawTextClipped (Text, false);
    if (Font == "small" || Font == "s0" || Font == "s1") H.Canvas.SetPos (X-1, Y+1); else H.Canvas.SetPos (X-2, Y+2);
    H.Canvas.DrawTextClipped (Text, false);
    if (Font == "small" || Font == "s0" || Font == "s1") H.Canvas.SetPos (X+1, Y-1); else H.Canvas.SetPos (X+2, Y-2);
    H.Canvas.DrawTextClipped (Text, false);
    if (Font == "small" || Font == "s0" || Font == "s1") H.Canvas.SetPos (X-1, Y-1); else H.Canvas.SetPos (X-2, Y-2);
    H.Canvas.DrawTextClipped (Text, false);
	H.Canvas.DrawColor = default.White;
    H.Canvas.SetPos (X, Y);
	H.Canvas.DrawTextClipped (Text, false);
  }
}

static simulated function TestDrawText (UTHUD H)
{
  // Position
  DrawText (H, 100, 100, "100.100");
  DrawText (H, 200, 200, "200.200");
  DrawText (H, 300, 300, "300.300");
  DrawText (H, 400, 400, "400.400");
  DrawText (H, 500, 500, "500.500");

  // Font
  DrawText (H, 650, 400, "small ABCDE 12345", "small");
  DrawText (H, 650, 450, "large ABCDE 12345", "large");
  DrawText (H, 650, 500, "s0 ABCDE 12345", "s0");
  DrawText (H, 650, 550, "s1 ABCDE 12345", "s1");
  DrawText (H, 650, 600, "s2 ABCDE 12345", "s2");

  // Style
  DrawText (H, 100, 400, "shadow ABCDE 12345", "s1", "goldshadow");
  DrawText (H, 100, 400, "shadow ABCDE 12345", "s1", "shadow");
  DrawText (H, 100, 450, "aura ABCDE 12345", "s1", "aura");

  // With tile
  DrawTile (H, "", 500, 100, 200, 100);
  DrawText (H, 510, 110, "Test 1 2 3", "s1", "white");  
}

// With multiple inheritance this might have been a method of UTHUD, UTCTFHUD, UTDuelHUD, UTOnslaughtHUD and UTTeamHUD
static simulated function BattleEnhanceHUD (UTHUD H)
{
  local BPRIDATA BD;

//  TestDrawText (H);
  BD = ReadBPRI(H.UTOwnerPRI);

  DrawText (H, 10, 500, "Level: "$BD.Level$" ("$BD.XP$"/"$BD.NeededXP$")");
  DrawText (H, 10, 530, "Mana: "$BD.Mana);
}

defaultproperties
{
  Black=(B=0,G=0,R=0,A=255)
  White=(B=255,G=255,R=255,A=255)
  Gold=(B=11,G=183,R=255,A=255)
  Name="BattleStaticFunctions"
}