/* 
    BattleRPG Copyright (C) 2007 Nico de Vries.

    This file is part of BattleRPG.

    BattleRPG is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see http://www.gnu.org/licenses/.
*/

class BattlePersistentPlayerDatabase extends Object config (BattlePersistentPlayerDatabase) dependson(BattleStaticFunctions);
// We want to keep BattleRPG settings and player data separated.

/*
  How does data get to (and from) BattlePersistentPlayerDatabase:
  - The BattleRPG mutator has a BPD variable which is initialized in InitMutator
  - ModifyPlayer calls LoadPlayerData to load data from disk in the proper PRI (player replication info) as BPRIDATA (special PRI data for RPG) 
  - LoadPlayerData always checks to see if the database is newer or older then what is in memory to prevent data loss
  - LoadPlayerData also stores a link to the BPD in the PRI
  - When WriteBPRI is called to change the BPRIDATA in the PRI it also calls SavePlayerData with this BPD
*/

struct RECORD {
  var name ID;
  var int Level; 
  var int XP;
  var int NeededXP;
};

var config array<RECORD> DATABASE;

function name PRI2Name (PlayerReplicationInfo PRI)
{
  return name (PRI.PlayerName);
//  return name (ID.Uid[0]$ID.Uid[1]$ID.Uid[2]$ID.Uid[3]$ID.Uid[4]$ID.Uid[5]$ID.Uid[6]$ID.Uid[7]);
} 

function Commit ()
{
  SaveConfig(); // Save the database to disk (useful if the server decides to crash)
}

function SavePlayerData (PlayerReplicationInfo PRI)
{
  local int index;
  local BPRIDATA BD;
  local RECORD R;

  BD = class'BattleStaticFunctions'.static.ReadBPRI (PRI);
LogInternal ("SavePlayerData "$PRI2Name (PRI));
  index = DATABASE.Find ('ID', PRI2NAME (PRI)); // Let's hope Epic made this with hashing
LogInternal ("SavePlayerData "$index);
  if (index==-1) { // insert new record
    R.ID = PRI2NAME (PRI);
    R.Level = BD.Level;
    R.XP = BD.XP;
    R.NeededXP = BD.NeededXP;
    DATABASE.AddItem (R);
  } else { // update existing record
    DATABASE[index].Level = BD.Level;
    DATABASE[index].XP = BD.XP;
    DATABASE[index].NeededXP = BD.NeededXP;
  } 
}

function LoadPlayerData (PlayerReplicationInfo PRI)
{
  local int index;
  local BPRIDATA BD;

LogInternal ("LoadPlayerData "$PRI2Name (PRI));
  index = DATABASE.Find ('ID', PRI2Name (PRI)); // Let's hope Epic made this with hashing
LogInternal ("LoadPlayerData "$index);

  BD = class'BattleStaticFunctions'.static.ReadBPRI (PRI);
  if (index != -1) {
    if (DATABASE[index].Level > BD.Level && DATABASE[index].XP > BD.XP) { // use the latest data
      BD.Level = DATABASE[index].Level;
      BD.XP = DATABASE[index].XP;
      BD.NeededXP = DATABASE[index].NeededXP;
    }
  }
  BD.BPD = self; // Make sure we can always reach BPD
  class'BattleStaticFunctions'.static.WriteBPRI (PRI, BD);
}

defaultproperties
{
  Name="BattlePersistentPlayerDatabase"
}