/* 
    BattleTRANS Copyright (C) 2007-2008 Nico de Vries.

    This file is part of BattleTRANS.

    BattleTRANS is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see http://www.gnu.org/licenses/.
*/

class BattleTRANS extends UTMutator config (BattleTRANS);

var config int HoverboardSpeed;
var config int TranslocatorSpeed;
var config int MaxFallSpeed;
var config string TranslocatorMode;
var config string LongTrail;

function PostBeginPlay()
{
  local UTGame G;
 
  Super.PostBeginPlay();
  G = UTGame(WorldInfo.Game);

  // Replace the translocator
  G.TranslocatorClass = class'BattleTRANS.BattleTranslocator';

  // Enable the hoverboard AND enable the translocator
  G.bAllowHoverboard = True;
  G.bAllowTranslocator = True;

  if (HoverboardSpeed   == 0) { HoverboardSpeed   = 150;  SaveConfig(); }
  if (TranslocatorSpeed == 0) { TranslocatorSpeed = 150;  SaveConfig(); }
  if (MaxFallSpeed      == 0) { MaxFallSpeed      = 150;  SaveConfig(); }
  if (TranslocatorMode  == "") { TranslocatorMode = "manualcam";  SaveConfig(); }
  if (LongTrail         == "") { LongTrail        = "on";  SaveConfig(); }

  SetTimer (5.0, True, 'RefreshEverything');
}

static final function string LCase (coerce string Text)
{
  local int IndexChar;
  for (IndexChar = 0; IndexChar < Len(Text); IndexChar++)
    if (Mid(Text, IndexChar, 1) >= "A" && Mid(Text, IndexChar, 1) <= "Z")
      Text = Left(Text, IndexChar) $ Chr(Asc(Mid(Text, IndexChar, 1)) + 32) $ Mid(Text, IndexChar + 1);
  return Text;
}

function Mutate (string MutateString, PlayerController Sender)
{
  if (Mid (LCase (MutateString), 0, 6) == "btset ") {
    if (Sender.PlayerReplicationInfo.bAdmin || Sender.WorldInfo.NetMode == NM_Standalone) {
      if (Mid (LCase (MutateString), 0, 21) == "btset hoverboardspeed") {
        HoverboardSpeed = int (Mid (MutateString, 22));
        if (HoverBoardSpeed < 10) HoverBoardSpeed = 10;
        if (HoverBoardSpeed > 1000) HoverBoardSpeed = 1000;
        UpdateSettingsEverywhere ();
        Sender.ClientMessage ("BattleTRANS: New HoverBoardSpeed is now:"@HoverBoardSpeed$"%");
      } else if (Mid (LCase (MutateString), 0, 23) == "btset translocatorspeed") {
        TranslocatorSpeed = int (Mid (MutateString, 24));
        if (TranslocatorSpeed < 10) TranslocatorSpeed = 10;
        if (TranslocatorSpeed > 1000) TranslocatorSpeed = 1000;
        UpdateSettingsEverywhere ();
        Sender.ClientMessage ("BattleTRANS: New TranslocatorSpeed is now:"@TranslocatorSpeed$"%");
      } else if (Mid (LCase (MutateString), 0, 18) == "btset maxfallspeed") {
        MaxFallSpeed = int (Mid (MutateString, 19));
        if (MaxFallSpeed < 10) MaxFallSpeed = 10;
        if (MaxFallSpeed > 1000) MaxFallSpeed = 1000;
        UpdateSettingsEverywhere ();
        Sender.ClientMessage ("BattleTRANS: New MaxFallSpeed is now:"@MaxFallSpeed$"%");
      } else if (LCase (MutateString) == "btset translocatormode off") {
        TranslocatorMode  = "off";     
        UpdateSettingsEverywhere ();
        Sender.ClientMessage ("BattleTRANS: New TranslocatorMode is now: off");
      } else if (LCase (MutateString) == "btset translocatormode standard") {
        TranslocatorMode  = "standard";
        UpdateSettingsEverywhere ();
        Sender.ClientMessage ("BattleTRANS: New TranslocatorMode is now: standard");
      } else if (LCase (MutateString) == "btset translocatormode autocam") {
        TranslocatorMode  = "autocam";
        UpdateSettingsEverywhere ();
        Sender.ClientMessage ("BattleTRANS: New TranslocatorMode is now: autocam");
      } else if (LCase (MutateString) == "btset translocatormode manualcam") {
        TranslocatorMode  = "manualcam";
        UpdateSettingsEverywhere ();
        Sender.ClientMessage ("BattleTRANS: New TranslocatorMode is now: manualcam");
      } else if (LCase (MutateString) == "btset longtrail on") {
        LongTrail  = "on";
        UpdateSettingsEverywhere ();
        Sender.ClientMessage ("BattleTRANS: LongTrail is now: on");
      } else if (LCase (MutateString) == "btset longtrail off") {
        LongTrail  = "off";
        UpdateSettingsEverywhere ();
        Sender.ClientMessage ("BattleTRANS: LongTrail is now: off");
      } else if (Mid (LCase (MutateString), 0, 16) == "btset btdefaults") {
        HoverboardSpeed   = 150;
        MaxFallSpeed      = 150; 
        TranslocatorSpeed = 150;
        TranslocatorMode  = "manualcam";
        LongTrail         = "on";
        UpdateSettingsEverywhere ();
        Sender.ClientMessage ("BattleTRANS: Default BattleTRANS settings");
      } else if (Mid (LCase (MutateString), 0, 17) == "btset ut3defaults") {
        HoverboardSpeed   = 100;
        MaxFallSpeed      = 100; 
        TranslocatorSpeed = 100;
        TranslocatorMode  = "standard";
        LongTrail         = "off";
        UpdateSettingsEverywhere ();
        Sender.ClientMessage ("BattleTRANS: default UT3 settings");
      } else {
        Sender.ClientMessage ("Unknown BattleTRANS btset command:"@MutateString);
      }            
    } else {
      Sender.ClientMessage ("You need to be administrator for that!");
    }
  }
  Super.Mutate(MutateString, Sender);
}

function UpdateSettingsEverywhere ()
{ 
  RefreshEverything ();
  SaveConfig ();
}

function RefreshEverything ()
{
  local Controller C;

  foreach WorldInfo.AllControllers(class'Controller', C) {
    UpdateSettings (C.PlayerReplicationInfo);
  }
}

function UpdateSettings (PlayerReplicationInfo PRI) 
{
  local BattlePRI BPRI;

  BPRI = GetBattlePRI (PRI);
  if (BPRI == None) { // No BPRI? Make one!
    AddBPRI (PRI);
    BPRI = GetBattlePRI (PRI);
    if (BPRI == None) return; // Give up
  }

  BPRI.HoverboardSpeed    = HoverboardSpeed;
  BPRI.TranslocatorSpeed  = TranslocatorSpeed;
  BPRI.MaxFallSpeed       = MaxFallSpeed;
  BPRI.TranslocatorMode   = TranslocatorMode;
  BPRI.LongTrail          = LongTrail;
}

simulated static function BattlePRI GetBattlePRI(PlayerReplicationInfo PRI)
{
  local UTLinkedReplicationInfo LPRI;

  if (UTPlayerReplicationInfo (PRI) == None) return None;
  if (UTPlayerReplicationInfo(PRI).CustomReplicationInfo == None) return None;
  for (LPRI = UTPlayerReplicationInfo(PRI).CustomReplicationInfo; LPRI != None; LPRI = LPRI.NextReplicationInfo) {
    if (BattlePRI (LPRI) != None) return BattlePRI (LPRI);
  }
  return None;
}

function AddBPRI (Actor Other)
{
  local BattlePRI BPRI;

  if (Role == ROLE_Authority) {
    if (UTPlayerReplicationInfo(Other)!=None) {
      if (GetBattlePRI (PlayerReplicationInfo(Other)) == None) {
        BPRI = Spawn (class'BattlePRI', Other);
        BPRI.NextReplicationInfo = UTPlayerReplicationInfo(Other).CustomReplicationInfo;
        UTPlayerReplicationInfo(Other).CustomReplicationInfo = BPRI;
      }
      UpdateSettings (PlayerReplicationInfo(Other));
    }
  }
}

function bool CheckReplacement(Actor Other)
{
  AddBPRI (Other);
  return super.CheckReplacement(Other);
}

simulated function ModifyPlayer(Pawn Other)
{
  local UTPawn P;
  
  if (Other != None) {
    P = UTPawn (Other);
    if (P != None) {
      P.MaxFallSpeed = 1250 * MaxFallSpeed / 100; // Protect the pawn from damage if landing after a translocation        
      P.HoverboardClass = Class'BattleTRANS.BattleBoard';
    }
    Other.CreateInventory (class'BattleTRANS.BattleInvItem4Switching');
  }
  Super.ModifyPlayer(Other);
}

defaultproperties
{
  Name="BattleTRANS"
}