class WalkerLegs extends KActor;

var float   DestroyTime, DestroyDelay;
var bool    InstantDestroy, AreSteady;

var enum L_State {
                  L_FirstStanding,
                  L_Standing,
                  L_Interpolating,
                  L_Free
                 } LegState[3];

var vector NewLocation[3], TargetLocation[3], Dist[3], Dist2D[3];
var float  Z_Buffer[3], kL[3], StepHeight[3], ArrivedDistance, StepSpeed, LegStepForce;
var Actor  Ground[3];
var float  StepSize, StepSoundVolume;

simulated function Tick(float dt)
{
 if (WalkerCraft(Owner) != none)
  {
   DestroyTime = Level.TimeSeconds + DestroyDelay;
   ProcessLegs(dt);
   KeepRotation();
  }
 else if ((DestroyTime < Level.TimeSeconds) || (InstantDestroy == true))
  Destroy();
}

function TakeDamage(int Damage, Pawn instigatedBy, Vector Hitlocation, Vector Momentum, class<DamageType> DamageType)
{
 if (WalkerCraft(Owner) != none)
   WalkerCraft(Owner).TakeDamage(Damage, instigatedBy, Hitlocation, Momentum, damageType);
}

// too complicated logic, no cycles here to be 200% sure
// loactions are always triangle
simulated function TraceParkedLocs(out vector TracedGround[3])
{
 local vector  TraceStarts[3], TraceEnds[3], HitNorm;
 local float   BestDist;
 local rotator Triangulator;

 BestDist = 64 + Sqrt(WalkerCraft(Owner).LegLength*WalkerCraft(Owner).LegLength - WalkerCraft(Owner).CurrentHeight*WalkerCraft(Owner).CurrentHeight)/2;
 // 0 - rear
 Triangulator.Yaw = - GetBoneRotation('Root').Yaw;
 TraceStarts[0] = GetBoneCoords('Root').Origin + vector(Triangulator)*BestDist;
 TraceEnds[0] = TraceStarts[0];
 TraceEnds[0].Z = TraceEnds[0].Z - WalkerCraft(Owner).LegLength*1.4;
 // 1 - left
 Triangulator.Yaw = GetBoneRotation('Root').Yaw - 10923;
 TraceStarts[1] = GetBoneCoords('Root').Origin + vector(Triangulator)*BestDist;
 TraceEnds[1] = TraceStarts[1];
 TraceEnds[1].Z = TraceEnds[1].Z - WalkerCraft(Owner).LegLength*1.4;
 // 2 - right
 Triangulator.Yaw = GetBoneRotation('Root').Yaw + 10923;
 TraceStarts[2] = GetBoneCoords('Root').Origin + vector(Triangulator)*BestDist;
 TraceEnds[2] = TraceStarts[2];
 TraceEnds[2].Z = TraceEnds[2].Z - WalkerCraft(Owner).LegLength*1.4;

 Ground[0] = Trace(TracedGround[0], HitNorm, TraceEnds[0], TraceStarts[0]);
 if ((Ground[0] == none) || (Ground[0].bWorldGeometry != true))
  TracedGround[0] = GetLegLocation(0);

 Ground[1] = Trace(TracedGround[1], HitNorm, TraceEnds[1], TraceStarts[1]);
 if ((Ground[1] == none) || (Ground[1].bWorldGeometry != true))
  TracedGround[1] = GetLegLocation(1);

 Ground[2] = Trace(TracedGround[2], HitNorm, TraceEnds[2], TraceStarts[2]);
 if ((Ground[2] == none) || (Ground[2].bWorldGeometry != true))
  TracedGround[2] = GetLegLocation(2);
}

simulated function byte ChooseLeg()
{
 local vector Destination;
 local float  RealOffSets[3];

 Destination = WalkerCraft(Owner).Location + WalkerCraft(Owner).LegLength*WalkerCraft(Owner).GetWalkDirection();
 RealOffsets[0] = VSize(Destination - GetLegLocation(0));
 RealOffsets[1] = VSize(Destination - GetLegLocation(1));
 RealOffsets[2] = VSize(Destination - GetLegLocation(2));
 if ((RealOffsets[0] >= RealOffsets[1]) && ((RealOffsets[0] >= RealOffsets[2])))
  return 0;
 else if (RealOffsets[1] >= RealOffsets[2])
  return 1;
 else
  return 2;
}

simulated function ProcessLegs(float dt)
{
 local byte CurrentLeg;

 for (CurrentLeg=0; CurrentLeg<3; CurrentLeg++)
  {
   if (LegState[CurrentLeg] == L_Standing)
    StandProcess(CurrentLeg);
   else if (LegState[CurrentLeg] == L_Interpolating)
    InterpolateProcess(dt, CurrentLeg);
   else if (LegState[CurrentLeg] == L_Free)
    {
     TraceGround();
     KAddImpulse(vect(0,0,-1000000)*dt+vect(0,0,-1), GetLegLocation(CurrentLeg), GetLegBoneName(CurrentLeg));
    }
  }
 if ((LegState[0] == L_Standing) && (LegState[1] == L_Standing) && (LegState[2] == L_Standing))
  AreSteady = true;
 else
  AreSteady = false;
}

simulated function TraceGround()
{
 local byte  MindLeg;

 for (MindLeg=0; MindLeg<3; MindLeg++)
  {
   if (FastTrace(GetLegLocation(MindLeg) - vect(0,0,32), GetLegLocation(MindLeg)) == false)
    SetLState(L_FirstStanding, MindLeg);
  }
}

simulated function SetLState(L_State NewState, byte Leg)
{
 if (NewState == L_Interpolating)
  {
   LegState[Leg] = L_Interpolating;
   NewLocation[Leg] = GetLegLocation(Leg);
   Z_Buffer[Leg] = GetLegLocation(Leg).Z;
   Dist[Leg] = TargetLocation[Leg] - GetLegLocation(Leg);
   Dist2D[Leg].X = Dist[Leg].X; Dist2D[Leg].Y = Dist[Leg].Y;
   kL[Leg] = VSize(Dist2D[Leg])/2;
   KSetFriction(0.5);
  }
 else  if (NewState == L_Standing)
  {
   LegState[Leg] = L_Standing;
   Z_Buffer[Leg] = GetLegLocation(Leg).Z;
  }
 else  if (NewState == L_FirstStanding)
  {
   LegState[Leg] = L_Standing;
   NewLocation[Leg] = GetLegLocation(Leg);
   Z_Buffer[Leg] = GetLegLocation(Leg).Z;
   TargetLocation[Leg] = GetLegLocation(Leg);
  }
}

simulated function InterpolateProcess(float dt, byte Leg)
{
 Dist[Leg] = TargetLocation[Leg] - NewLocation[Leg];
 Dist2D[Leg].X = Dist[Leg].X; Dist2D[Leg].Y = Dist[Leg].Y;
 NewLocation[Leg].X = NewLocation[Leg].X + Normal(Dist[Leg]).X*dt*StepSpeed;
 NewLocation[Leg].Y = NewLocation[Leg].Y + Normal(Dist[Leg]).Y*dt*StepSpeed;
 Z_Buffer[Leg] = Z_Buffer[Leg] + Normal(Dist[Leg]).Z*dt*StepSpeed;
 NewLocation[Leg].Z = Z_Buffer[Leg] + StepHeight[Leg]*(1 - abs(1 - VSize(Dist2D[Leg])/kL[Leg]));
  KAddImpulse(LegStepForce*(NewLocation[Leg] - GetLegLocation(Leg))+vect(0,0,-1), GetLegLocation(Leg), GetLegBoneName(Leg));
 if(VSize(Dist[Leg]) < ArrivedDistance)
  {
   NewLocation[Leg] = TargetLocation[Leg];
   SetLState(L_Standing, Leg);
   PlaySound( Sound'DW_S.tento_step',SLOT_None,255*StepSoundVolume,,128,64,false);
   WalkerCraft(Owner).NextStepTime = Level.TimeSeconds;
  }
}

simulated function StandProcess(byte Leg)
{
 if (VSize(NewLocation[Leg] - GetLegLocation(Leg)) > ArrivedDistance)
  KAddImpulse(LegStepForce*(NewLocation[Leg] - GetLegLocation(Leg))+vect(0,0,-1), GetLegLocation(Leg), GetLegBoneName(Leg));
}

simulated function KeepRotation()
{
 local float  MasterZ;
 local vector ShouldersZ[3];
 local vector LonelyImpulce;

 MasterZ = GetBoneCoords('Root').Origin.Z;
 ShouldersZ[0] = GetBoneCoords('Leg1_Bone2').Origin;
 ShouldersZ[1] = GetBoneCoords('Leg2_Bone2').Origin;
 ShouldersZ[2] = GetBoneCoords('Leg3_Bone2').Origin;

 if (ShouldersZ[0].Z > MasterZ)
  {
   LonelyImpulce.Z = (MasterZ - ShouldersZ[0].Z)*10000 + 1;
   KAddImpulse(LonelyImpulce, ShouldersZ[0], 'Leg1_Bone2');
  }
 if (ShouldersZ[1].Z > MasterZ)
  {
   LonelyImpulce.Z = (MasterZ - ShouldersZ[1].Z)*10000 + 1;
   KAddImpulse(LonelyImpulce, ShouldersZ[1], 'Leg2_Bone2');
  }
 if (ShouldersZ[2].Z > MasterZ)
  {
   LonelyImpulce.Z = (MasterZ - ShouldersZ[2].Z)*10000 + 1;
   KAddImpulse(LonelyImpulce, ShouldersZ[2], 'Leg3_Bone2');
  }
}

simulated function StepTo(vector StepToLoc, float CustomStepHeight, byte Leg)
{
 local float Distance;

 Distance = VSize(GetLegLocation(Leg) - StepToLoc);
 StepSoundVolume = Distance/StepSize;
 StepHeight[Leg] = CustomStepHeight;
 StepSpeed = Distance*2;
 TargetLocation[Leg] = StepToLoc;
 SetLState(L_Interpolating, Leg);
}

simulated function vector GetMidPoint()
{
 return ( GetBoneCoords('Leg1_End').Origin +
          GetBoneCoords('Leg2_End').Origin +
          GetBoneCoords('Leg3_End').Origin )/3;
}

simulated function vector GetLegLocation(byte NewLeg)
{
 if (NewLeg == 0)
  return GetBoneCoords('Leg1_End').Origin;
 else if (NewLeg == 1)
  return GetBoneCoords('Leg2_End').Origin;
 else if (NewLeg == 2)
  return GetBoneCoords('Leg3_End').Origin;
}

simulated function float GetLegStretch(byte NewLeg)
{
 if (NewLeg == 0)
  return VSize(GetBoneCoords('Leg1_End').Origin - GetBoneCoords('Root').Origin)/WalkerCraft(Owner).LegLength;
 else if (NewLeg == 1)
  return VSize(GetBoneCoords('Leg2_End').Origin - GetBoneCoords('Root').Origin)/WalkerCraft(Owner).LegLength;
 else if (NewLeg == 2)
  return VSize(GetBoneCoords('Leg3_End').Origin - GetBoneCoords('Root').Origin)/WalkerCraft(Owner).LegLength;
}

simulated function name GetLegBoneName(byte NewLeg)
{
 if (NewLeg == 0)
  return 'Leg1_End';
 else if (NewLeg == 1)
  return 'Leg2_End';
 else if (NewLeg == 2)
  return 'Leg3_End';
}

defaultproperties
{
     DestroyDelay=3.000000
     LegState(0)=L_Free
     LegState(1)=L_Free
     LegState(2)=L_Free
     ArrivedDistance=6.000000
     StepSpeed=256.000000
     LegStepForce=2000.000000
     StepSize=512.000000
     DrawType=DT_Mesh
     bUseDynamicLights=True
     bNoDelete=False
     Physics=PHYS_KarmaRagDoll
     Mesh=SkeletalMesh'DW_A.Legs'
     Begin Object Class=KarmaParamsSkel Name=KarmaParamsSkel0
         KSkeleton="DW_Legs"
         KLinearDamping=5.000000
         KAngularDamping=5.000000
         KStartEnabled=True
         KActorGravScale=-0.100000
         bKDoubleTickRate=True
         bKImportantRagdoll=True
         KFriction=1.000000
     End Object
     KParams=KarmaParamsSkel'UT3DarkWalker.WalkerLegs.KarmaParamsSkel0'

}
